package com.upyuns.platform.rs.website.biz;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSONObject;
import com.github.wxiaoqi.security.admin.feign.UserFeign;
import com.github.wxiaoqi.security.admin.feign.dto.AppUserDTO;
import com.github.wxiaoqi.security.common.msg.ObjectRestResponse;
import com.github.wxiaoqi.security.common.util.ClientUtil;
import com.github.wxiaoqi.security.common.util.OrderUtil;
import com.github.wxiaoqi.security.common.util.Query;
import com.github.wxiaoqi.security.common.util.UUIDUtils;
import com.github.wxiaoqi.security.common.util.process.ResultCode;
import com.github.wxiaoqi.security.common.util.result.JsonResultUtil;
import com.github.wxiaoqi.security.common.vo.PageDataVO;
import com.upyuns.platform.rs.datacenter.fegin.DatacenterFeign;
import com.upyuns.platform.rs.datacenter.pojo.ImageDataVO;
import com.upyuns.platform.rs.datacenter.pojo.ImagePageFeginDTO;
import com.upyuns.platform.rs.universal.constant.DictionaryKey;
import com.upyuns.platform.rs.universal.entity.Dictionary;
import com.upyuns.platform.rs.universal.entity.OrderPay;
import com.upyuns.platform.rs.universal.feign.ThirdFeign;
import com.upyuns.platform.rs.universal.vo.OrderPayVo;
import com.upyuns.platform.rs.website.dto.*;
import com.upyuns.platform.rs.website.entity.*;
import com.upyuns.platform.rs.website.mapper.OrderItemMapper;
import com.upyuns.platform.rs.website.type.*;
import com.upyuns.platform.rs.website.vo.ItemInfoVo;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.regexp.RE;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import com.upyuns.platform.rs.website.mapper.OrderInfoMapper;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import tk.mybatis.mapper.entity.Example;

import javax.servlet.http.HttpServletRequest;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.stream.Collectors;

/**
 * 订单信息
 *
 * @author jiaoruizhen
 * @email jiaoruizhen@126.com
 * @date 2020-12-03 13:29:23
 */
@Service
@Slf4j
public class OrderInfoBiz extends BaseBiz<OrderInfoMapper, OrderInfo> {

	@Autowired
	OrderItemBiz orderItemBiz;
	@Autowired
	IndustryApplicationInfoBiz industryApplicationInfoBiz;

	@Autowired
	ImageImgStorageBiz imageImgStorageBiz;

	@Autowired
	ShoppingCartInfoBiz shoppingCartInfoBiz;

	@Autowired
	ImageInfoRelationBiz imageInfoRelationBiz;

	@Autowired
	ThirdFeign thirdFeign;

	@Autowired
	UserFeign userFeign;

	@Autowired
	HttpServletRequest request;

	@Autowired
	UserAddressBiz userAddressBiz;

	@Autowired
	OrderEInvoiceBiz orderEInvoiceBiz;

	@Autowired
	UserInvoiceBiz userInvoiceBiz;

	@Autowired
	DatacenterFeign datacenterFeign;

	@Autowired
	CustomFormBiz customFormBiz;

	@Autowired
	NoticeInfoBiz noticeInfoBiz;

	/**
	 * 添加订单
	 *
	 * @param orderInfo
	 * @return
	 */
	public ObjectRestResponse addUpdate(OrderInfo orderInfo) {
		if (orderInfo == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		if (orderInfo.getOrderId() == null) {
			insertSelectiveRe(orderInfo);
		} else {
			OrderInfo old = selectById(orderInfo.getOrderId());
			BeanUtil.copyProperties(orderInfo, old, CopyOptions.create().setIgnoreNullValue(true).setIgnoreError(true));
			updateSelectiveByIdRe(old);
		}
		return ObjectRestResponse.succ();
	}

	/**
	 * 下单
	 *
	 * @param orderInfo
	 * @return
	 */
	public ObjectRestResponse generateOrder(OrderInfo orderInfo) {
		if (orderInfo == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		if (StringUtils.isBlank(orderInfo.getOrderNo())) {
			orderInfo.setOrderNo(OrderUtil.GetOrderNumber("", OrderUtil.APP_MID));
		}
		OrderInfo old = getOneByOrderNo(orderInfo.getOrderNo());
		while (old != null) {
			orderInfo.setOrderNo(OrderUtil.GetOrderNumber("", OrderUtil.APP_MID));
			old = getOneByOrderNo(orderInfo.getOrderNo());
		}
		orderInfo.setPayCode(UUIDUtils.genCodes(0));
		return addUpdate(orderInfo);
	}

	/**
	 * 根据订单号查询订单信息
	 *
	 * @param orderNo
	 * @return
	 */
	public OrderInfo getOneByOrderNo(String orderNo) {
		Example example = new Example(OrderInfo.class);
		example.createCriteria().andEqualTo("orderNo", orderNo).andNotEqualTo("status", -1).andNotEqualTo("status", -2);
		return mapper.selectOneByExample(example);
	}

	/**
	 * 查询用户所有订单
	 *
	 * @param userOrderDto
	 * @return
	 */
	public ObjectRestResponse getUserOrder(UserOrderDto userOrderDto) {
		Query query = new Query(userOrderDto);
		Example example = new Example(OrderInfo.class);
		Example.Criteria orderInfoCriteria = example.createCriteria();
		if (userOrderDto.getSatellite() != null) {
			orderInfoCriteria.andLike("satellite", "%"+userOrderDto.getSatellite() +"%");
		}
		if (userOrderDto.getStatus() != null) {
			orderInfoCriteria.andEqualTo("status", userOrderDto.getStatus());
		}
		if (userOrderDto.getNewStatus() != null) {
			orderInfoCriteria.andEqualTo("status", userOrderDto.getNewStatus());
		}
		if (userOrderDto.getPayStatus() != null) {
			orderInfoCriteria.andEqualTo("payStatus", userOrderDto.getPayStatus());
		}
		if (userOrderDto.getUserId() != null) {
			orderInfoCriteria.andEqualTo("userId", userOrderDto.getUserId());
		}
		if (StringUtils.isNotBlank(userOrderDto.getStartTime()) && StringUtils.isNotBlank(userOrderDto.getEndTime())) {
			orderInfoCriteria.andBetween("crtTime", userOrderDto.getStartTime(), userOrderDto.getEndTime());
		}
		if (StringUtils.isNotBlank(userOrderDto.getKeywords())) {
			orderInfoCriteria.andLike("realName", "%" + userOrderDto.getKeywords() + "%")
					.orLike("realName", "%" + userOrderDto.getKeywords() + "%")
					.orLike("phone", "%" + userOrderDto.getKeywords() + "%")
					.orLike("orderNo", "%" + userOrderDto.getKeywords() + "%");
		}

		orderInfoCriteria.andNotEqualTo("status", -1);
		example.orderBy("crtTime").desc();
		PageDataVO<OrderInfo> pageDataVO = PageDataVO.pageInfo(query, () -> mapper.selectByExample(example));

		if (pageDataVO != null && pageDataVO.getData() != null && pageDataVO.getData().size() > 0) {
			fillOrderItem(pageDataVO.getData());
			fillOrderEInvoice(pageDataVO.getData());
			return ObjectRestResponse.succ(pageDataVO);
		}
		return ObjectRestResponse.succ(pageDataVO);
	}

	/**
	 * 填充订单发票信息
	 *
	 * @param orderInfoList
	 */
	private void fillOrderEInvoice(List<OrderInfo> orderInfoList) {
		List<Integer> orderInvoiceIdList = orderInfoList.stream().map(OrderInfo::getInvoiceId).collect(Collectors.toList());
		if (orderInvoiceIdList != null && orderInvoiceIdList.size() > 0) {
			List<OrderEInvoice> orderEInvoiceList = orderEInvoiceBiz.getOrderEInvoice(orderInvoiceIdList);
			if (orderEInvoiceList != null && orderEInvoiceList.size() > 0) {
				Map<Integer, List<OrderEInvoice>> map = orderEInvoiceList.stream().collect(Collectors.groupingBy(OrderEInvoice::getId));
				if (map != null && map.size() > 0) {
					orderInfoList.stream().forEach(orderInfo -> {
						if (map.get(orderInfo.getInvoiceId()) != null && orderInfo != null) {
							orderInfo.setOrderEInvoice(map.get(orderInfo.getInvoiceId()).get(0));
						}
					});
				}
			}
		}
	}

	/**
	 * 查询订单详情
	 *
	 * @param orderId
	 * @return
	 */
	public OrderInfo getOrderDetail(Long orderId) {
		Example example = new Example(OrderInfo.class);
		example.createCriteria().andEqualTo("orderId", orderId).andNotEqualTo("status", -1).andNotEqualTo("status", -2);
		OrderInfo orderInfo = mapper.selectOneByExample(example);
		if (orderInfo == null) {
			return null;
		}
		if (orderInfo.getInvoiceId() != null) {
			OrderEInvoice orderEInvoice = orderEInvoiceBiz.selectById(orderInfo.getInvoiceId());
			if (orderEInvoice != null) {
				orderInfo.setOrderEInvoice(orderEInvoice);
			}
		}
		List<OrderItem> orderItemList = orderItemBiz.getByOrderId(orderInfo.getOrderId());
		if (orderItemList != null && orderItemList.size() > 0) {
			orderInfo.setOrderItemList(orderItemList);
		}
		return orderInfo;
	}

	/**
	 * 填充订单商品
	 *
	 * @param orderInfoList
	 */
	private void fillOrderItem(List<OrderInfo> orderInfoList) {
		List<Long> orderIdList = orderInfoList.stream().map(OrderInfo::getOrderId).collect(Collectors.toList());
		List<OrderItem> orderItemList = orderItemBiz.getAllByOrderId(orderIdList);
		if (orderItemList != null) {
			Map<Long, List<OrderItem>> map = orderItemList.stream().collect(Collectors.groupingBy(OrderItem::getOrderId));
			if (map != null && map.size() > 0) {
				orderInfoList.stream().forEach(orderInfo -> {
					orderInfo.setOrderItemList(map.get(orderInfo.getOrderId()));
				});
			}
		}
	}

	/**
	 * 提交订单
	 *
	 * @param orderInfoDto
	 * @return
	 */
	public ObjectRestResponse submitOrder(OrderInfoDto orderInfoDto) {
		AppUserDTO appUserDTO = userFeign.userDetailByToken(request.getHeader("Authorization")).getData();
		if (appUserDTO == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.RSTOKEN_EXPIRED_CODE, ResultCode.getMsg(ResultCode.RSTOKEN_EXPIRED_CODE));
		}

		if (orderInfoDto == null || orderInfoDto.getItemInfoDtoList() == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		List<ItemInfoDto> itemInfoDtoList = orderInfoDto.getItemInfoDtoList();
		List<OrderItem> orderItemList = new ArrayList<>();
		AtomicReference<Integer> number = new AtomicReference<>(0);
		AtomicReference<Double> amount = new AtomicReference<>((double) 0);
		StringBuilder sb = new StringBuilder();
		itemInfoDtoList.forEach(itemInfoDto -> {
			switch (OrderTypeEnum.getByCode(itemInfoDto.getType())) {
				//标准数据
				case STANDARD_DATA:
					List<ImageDataVO> imageDataVOList = datacenterFeign.queryByIds(itemInfoDto.getId() + "").getData();
					if (imageDataVOList != null && imageDataVOList.size() > 0) {
						ImageDataVO imageDataVO = imageDataVOList.get(0);
						OrderItem orderItem = new OrderItem();
						orderItem.setType(itemInfoDto.getType());
						orderItem.setTotalAmount(imageDataVO.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber())));
						orderItem.setPrice(imageDataVO.getPrice());
						orderItem.setNumber(itemInfoDto.getNumber());
						orderItem.setDetailJson(JSONObject.toJSONString(imageDataVO));
						orderItem.setDetailId(Long.valueOf(imageDataVO.getId()));
						orderItemList.add(orderItem);
						sb.append(imageDataVO.getImageSatelliteType());
						sb.append(",");
						//总数量
						number.updateAndGet(v -> v + itemInfoDto.getNumber());
						BigDecimal totalAmount = imageDataVO.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber()));
						amount.updateAndGet(v -> new Double((double) (v + totalAmount.doubleValue())));
					}
					break;
				//行业应用
				case INDUSTRY_INFO:
					IndustryApplicationInfo industryApplicationInfo = industryApplicationInfoBiz.getOneById(itemInfoDto.getId().intValue()).getData();
					if (industryApplicationInfo != null) {
						OrderItem orderItem = new OrderItem();
						orderItem.setType(itemInfoDto.getType());
						orderItem.setTotalAmount(industryApplicationInfo.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber())));
						orderItem.setPrice(industryApplicationInfo.getPrice());
						orderItem.setNumber(itemInfoDto.getNumber());
						orderItem.setName(industryApplicationInfo.getTitle());
						orderItem.setItemPic(industryApplicationInfo.getCoverImg());
						orderItem.setDetailJson(JSONObject.toJSONString(industryApplicationInfo));
						orderItem.setDetailId(Long.valueOf(industryApplicationInfo.getId()));
						orderItemList.add(orderItem);
						//总数量
						number.updateAndGet(v -> v + itemInfoDto.getNumber());
						BigDecimal totalAmount = industryApplicationInfo.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber()));
						amount.updateAndGet(v -> new Double((double) (v + totalAmount.doubleValue())));
					}
					break;
				//影像图库
				case IMAGE_STORAGE:
					ImageInfoRelation imageInfoRelation = imageInfoRelationBiz.selectById(itemInfoDto.getId());
					if (imageInfoRelation != null) {
						ImageImgStorage imageImgStorage = imageImgStorageBiz.getDetail(itemInfoDto.getId().intValue()).getData();
						if (imageImgStorage != null) {
							List<ImageInfoRelation> imageInfoRelationList = new ArrayList<>();
							imageInfoRelationList.add(imageInfoRelation);
							imageImgStorage.setImageInfoRelationList(imageInfoRelationList);
							OrderItem orderItem = new OrderItem();
							orderItem.setType(itemInfoDto.getType());
							orderItem.setTotalAmount(imageInfoRelation.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber())));
							orderItem.setPrice(imageInfoRelation.getPrice());
							orderItem.setNumber(itemInfoDto.getNumber());
							orderItem.setName(imageImgStorage.getName() + " (" + imageInfoRelation.getFileWidth()+ ")");
							orderItem.setItemPic(imageImgStorage.getCoverImg());
							orderItem.setDetailJson(JSONObject.toJSONString(imageImgStorage));
							orderItem.setDetailId(Long.valueOf(imageImgStorage.getId()));
							orderItemList.add(orderItem);
							//总数量
							number.updateAndGet(v -> v + itemInfoDto.getNumber());
							BigDecimal totalAmount = imageInfoRelation.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber()));
							amount.updateAndGet(v -> new Double((double) (v + totalAmount.doubleValue())));
						}
					}
					break;
				//购物车
				case SHOPPING_CART_INFO:
					ShoppingCartInfo shoppingCartInfo = shoppingCartInfoBiz.selectById(itemInfoDto.getId());
					if (shoppingCartInfo != null) {
						OrderItem orderItem = new OrderItem();
						orderItem.setType(shoppingCartInfo.getType());
						orderItem.setTotalAmount(shoppingCartInfo.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber())));
						orderItem.setPrice(shoppingCartInfo.getPrice());
						orderItem.setNumber(itemInfoDto.getNumber());
						orderItem.setName(shoppingCartInfo.getName());
						orderItem.setItemPic(shoppingCartInfo.getItemPic());
						orderItem.setDetailJson(shoppingCartInfo.getDetailJson());
						orderItem.setDetailId(shoppingCartInfo.getDetailId());
						orderItemList.add(orderItem);
						number.updateAndGet(v -> v + itemInfoDto.getNumber());
						BigDecimal totalAmount = shoppingCartInfo.getTotalAmount();
						amount.updateAndGet(v -> new Double((double) (v + totalAmount.doubleValue())));
						//下单后删除购物车
						shoppingCartInfo.setIsDel(1);
						shoppingCartInfoBiz.delete(shoppingCartInfo);
					}
					break;
				default:
					break;
			}
		});

		OrderInfo orderInfo = new OrderInfo();
		orderInfo.setRealName(appUserDTO.getRealname());
		orderInfo.setUserId(appUserDTO.getUserid());
		orderInfo.setPhone(appUserDTO.getUsername());
		orderInfo.setNickname(appUserDTO.getNickname());
		orderInfo.setHeadImg(appUserDTO.getHeadimgurl());
		if (orderInfoDto.getUserAddressId() != null) {
			UserAddress userAddress = userAddressBiz.selectById(orderInfoDto.getUserAddressId());
			if (userAddress != null) {
				orderInfo.setReceiveName(userAddress.getConsigneeName());
				orderInfo.setReceivePhone(userAddress.getConsigneePhone());
				orderInfo.setReceiveAddress(userAddress.getProvince() + userAddress.getCity() + userAddress.getTown() + userAddress.getConsigneeAddress());
			}
		}

		orderInfo.setRemark(orderInfoDto.getRemark());
		orderInfo.setAmount(new BigDecimal(amount.get()));
		orderInfo.setNumber(number.get());
		orderInfo.setTotalAmount(orderInfo.getAmount());
		if (orderInfo.getAmount().compareTo(BigDecimal.ZERO) < 0 || orderInfo.getTotalAmount().compareTo(BigDecimal.ZERO) < 0) {
			return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单金额不能为0");
		}
		if (orderInfo.getTotalAmount().compareTo(BigDecimal.ZERO) ==  0) {
			orderInfo.setStatus(OrderStatusEnum.ORDER_SEND.getType());
			orderInfo.setPayStatus(2);
		}

		//订单发票信息
		if (orderInfoDto.getUserInvoiceId() != null) {
			UserInvoice userInvoice = userInvoiceBiz.selectById(orderInfoDto.getUserInvoiceId());
			if (userInvoice != null) {
				OrderEInvoice orderEInvoice = new OrderEInvoice();
				orderEInvoice.setAmount(orderInfo.getAmount());
				orderEInvoice.setType(1);
				orderEInvoice.setOrderIds(orderInfo.getOrderId() + "");
				orderEInvoice.setTitleType(userInvoice.getTitleType());
				orderEInvoice.setTitleName(userInvoice.getTitleName());
				orderEInvoice.setTaxCode(userInvoice.getTaxCode());
				orderEInvoice.setContentRemark(orderInfoDto.getRemark());
				orderEInvoice.setContentAccount(userInvoice.getOpenBank() + "  " + userInvoice.getAccount());
				orderEInvoice.setContentContact(orderInfo.getReceiveAddress() + "  " + orderInfo.getReceiveName() + "  " + orderInfo.getReceivePhone());
				//发票内容
				if (StringUtils.isNotBlank(orderInfoDto.getInvoiceContent())) {
					orderEInvoice.setContentTitle(orderInfoDto.getInvoiceContent());
				}
				orderEInvoiceBiz.addUpdate(orderEInvoice);
				orderInfo.setInvoiceStatus(1);
				orderInfo.setInvoiceId(orderEInvoice.getId());
				Dictionary dictionary = thirdFeign.findDictionaryByTypeAndCode(DictionaryKey.APP_ORDER, DictionaryKey.SEND_FEE);
				if (dictionary != null && StringUtils.isNotBlank(dictionary.getDetail())) {
					orderInfo.setFee(new BigDecimal(dictionary.getDetail()));
				}
				if (orderInfo.getFee() != null) {
					orderInfo.setTotalAmount(orderInfo.getAmount().add(orderInfo.getFee()));
				}
			}
		}

		orderInfo.setSendType(orderInfoDto.getSendType());
		orderInfo.setType(1);
		orderInfo.setSatellite(sb.toString());
		/**
		 * 新增订单
		 */
		orderInfo.setOrderNo(OrderUtil.GetOrderNumber("", OrderUtil.APP_MID));
		generateOrder(orderInfo);
		orderItemList.stream().forEach(orderItem -> {
			if (orderItem != null) {
				orderItem.setOrderId(orderInfo.getOrderId());
				orderItemBiz.insertSelective(orderItem);
			}
		});
		//添加通知信息
		noticeInfoBiz.addObj("订单" + orderInfo.getOrderNo() + "已创建，待支付", NoticeTypeEnum.ORDER_INFO.getType(), appUserDTO.getUserid(), String.valueOf(orderInfo.getOrderId()));
		return ObjectRestResponse.succ(orderInfo);
	}

	/**
	 * 确认收货
	 *
	 * @return
	 */
	public ObjectRestResponse confirmReceived(Long orderId) {
		if (orderId == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(orderId);
		if (orderInfo == null || orderInfo.getStatus().equals(OrderStatusEnum.ORDER_DELETE.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		if (!orderInfo.getStatus().equals(OrderStatusEnum.ORDER_SEND.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单状态错误");
		}
		orderInfo.setShippingStatus(OrderShippingStatusEnum.ORDER_RECEIVED.getType());
		orderInfo.setStatus(OrderStatusEnum.ORDER_RECEIVED.getType());
		updateSelectiveByIdRe(orderInfo);
		return ObjectRestResponse.succ();
	}

	/**
	 * 发货
	 *
	 * @return
	 */
	public ObjectRestResponse orderSend(ConfirmReceivedDto confirmReceivedDto) {
		if (confirmReceivedDto == null || confirmReceivedDto.getOrderId() == null || StringUtils.isBlank(confirmReceivedDto.getTrackingNumber())) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(confirmReceivedDto.getOrderId());
		if (orderInfo == null || orderInfo.getStatus().equals(OrderStatusEnum.ORDER_DELETE.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		if (!orderInfo.getStatus().equals(OrderStatusEnum.ORDER_PAID.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单状态错误");
		}
		orderInfo.setShippingStatus(OrderShippingStatusEnum.ORDER_SEND.getType());
		orderInfo.setStatus(OrderStatusEnum.ORDER_SEND.getType());
		orderInfo.setTrackingNumber(confirmReceivedDto.getTrackingNumber());
		orderInfo.setTrackingImg(confirmReceivedDto.getTrackingImg());
		updateSelectiveByIdRe(orderInfo);
		//添加通知信息
		noticeInfoBiz.addObj("订单" + orderInfo.getOrderNo() + "已发货", NoticeTypeEnum.ORDER_INFO.getType(), orderInfo.getUserId(), String.valueOf(orderInfo.getOrderId()));
		return ObjectRestResponse.succ();
	}

	/**
	 * 开发票
	 *
	 * @return
	 */
	public ObjectRestResponse setInvoice(OrderInvoiceDto orderInvoiceDto) {
		if (orderInvoiceDto == null || orderInvoiceDto.getOrderId() == null || StringUtils.isBlank(orderInvoiceDto.getInvoiceUrl())) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(orderInvoiceDto.getOrderId());
		if (orderInfo == null || orderInfo.getStatus().equals(OrderStatusEnum.ORDER_DELETE.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		if (orderInfo.getInvoiceId() != null) {
			OrderEInvoice orderEInvoice = orderEInvoiceBiz.selectById(orderInfo.getInvoiceId());
			if (orderEInvoice == null) {
				return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单发票不存在");
			}
			orderEInvoice.setStatus(OrderInvoiceStatusEnum.INVOICE_OPEN.getType());
			orderEInvoice.setInvoiceUrl(orderInvoiceDto.getInvoiceUrl());
			orderEInvoiceBiz.updateSelectiveByIdRe(orderEInvoice);

			orderInfo.setInvoiceStatus(OrderInvoiceStatusEnum.INVOICE_OPEN.getType());
			updateSelectiveByIdRe(orderInfo);
			//添加通知信息
			noticeInfoBiz.addObj("订单" + orderInfo.getOrderNo() + "已开票", NoticeTypeEnum.ORDER_INFO.getType(), orderInfo.getUserId(), String.valueOf(orderInfo.getOrderId()));

		} else {
			return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单未开票");
		}
		return ObjectRestResponse.succ();
	}

	/**
	 * 发票发货
	 *
	 * @param confirmReceivedDto
	 * @return
	 */
	public ObjectRestResponse invoiceSend(ConfirmReceivedDto confirmReceivedDto) {
		if (confirmReceivedDto == null || confirmReceivedDto.getOrderId() == null || StringUtils.isBlank(confirmReceivedDto.getTrackingNumber())) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(confirmReceivedDto.getOrderId());
		if (orderInfo == null || orderInfo.getStatus().equals(OrderStatusEnum.ORDER_DELETE.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}

		if (!orderInfo.getInvoiceStatus().equals(OrderInvoiceStatusEnum.INVOICE_NO_OPEN.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单状态错误");
		}

		OrderEInvoice orderEInvoice = orderEInvoiceBiz.selectById(orderInfo.getInvoiceId());
		if (orderEInvoice == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单发票不存在");
		}
		orderEInvoice.setTrackingNumber(confirmReceivedDto.getTrackingNumber());
		orderEInvoice.setTrackingImg(confirmReceivedDto.getTrackingImg());
		orderEInvoiceBiz.updateSelectiveByIdRe(orderEInvoice);

		orderInfo.setInvoiceStatus(OrderInvoiceStatusEnum.INVOICE_SEND.getType());
		updateSelectiveByIdRe(orderInfo);
		//添加通知信息
		noticeInfoBiz.addObj("订单" + orderInfo.getOrderNo() + "发票已发货", NoticeTypeEnum.ORDER_INFO.getType(), orderInfo.getUserId(), String.valueOf(orderInfo.getOrderId()));
		return ObjectRestResponse.succ();
	}

	/**
	 * 订单发票确认收货
	 *
	 * @param orderId
	 * @return
	 */
	public ObjectRestResponse confirmOrderInvoice(Long orderId) {
		if (orderId == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(orderId);
		if (orderInfo == null || orderInfo.getStatus().equals(OrderStatusEnum.ORDER_DELETE.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}

		if (!orderInfo.getInvoiceStatus().equals(OrderInvoiceStatusEnum.INVOICE_SEND.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单状态错误");
		}
		orderInfo.setInvoiceStatus(OrderInvoiceStatusEnum.INVOICE_RECEIVED.getType());
		updateSelectiveByIdRe(orderInfo);
		return ObjectRestResponse.succ();
	}

	/**
	 * 取消订单
	 *
	 * @param param
	 * @return
	 */
	public ObjectRestResponse cancelOrder(OrderInfo param) {
		if (param == null || param.getOrderId() == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(param.getOrderId());
		if (orderInfo == null || orderInfo.getStatus().equals(OrderStatusEnum.ORDER_DELETE.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		//订单状态为待支付前, 直接改变状态
		if (orderInfo.getStatus() < 3) {
			orderInfo.setStatus(OrderStatusEnum.ORDER_CANCEL.getType());
		} else if (orderInfo.getStatus() == 3) { //订单已支付
			orderInfo.setStatus(OrderStatusEnum.ORDER_CANCEL.getType());
			//todo 执行退款操作

		} else if (orderInfo.getStatus() > 3) { //已发货订单不可取消
			return ObjectRestResponse.createFailedResult(ResultCode.FAILED_CODE, "订单不可取消");
		}
		if (StringUtils.isNotBlank(param.getRemark())) {
			orderInfo.setCancelRemark(param.getRemark());
		}
		updateSelectiveByIdRe(orderInfo);
		return ObjectRestResponse.succ();
	}

	/**
	 * 删除订单
	 *
	 * @param orderId
	 * @return
	 */
	public ObjectRestResponse deleteOrder(Long orderId) {
		if (orderId == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(orderId);
		if (orderInfo == null || orderInfo.getStatus().equals(OrderStatusEnum.ORDER_DELETE.getType())) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		orderInfo.setStatus(OrderStatusEnum.ORDER_DELETE.getType());
		updateSelectiveByIdRe(orderInfo);
		return ObjectRestResponse.succ();
	}


	/**
	 * 查询用户所有订单
	 *
	 * @param userOrderDto
	 * @return
	 */
	public ObjectRestResponse getOrderList(UserOrderDto userOrderDto) {
		Query query = new Query(userOrderDto);
		Example example = new Example(OrderInfo.class);
		Example.Criteria orderInfoCriteria = example.createCriteria();
		if (userOrderDto.getStatus() != null) {
			orderInfoCriteria.andEqualTo("status", userOrderDto.getStatus());
		}
		if (userOrderDto.getPayStatus() != null) {
			orderInfoCriteria.andEqualTo("payStatus", userOrderDto.getPayStatus());
		}
		if (userOrderDto.getUserId() != null) {
			orderInfoCriteria.andEqualTo("userId", userOrderDto.getUserId());
		}
		if (StringUtils.isNotBlank(userOrderDto.getOrderNo())) {
			orderInfoCriteria.andEqualTo("orderNo", userOrderDto.getOrderNo());
		}
		if (StringUtils.isNotBlank(userOrderDto.getNickname())) {
			orderInfoCriteria.andLike("nickname", "%" + userOrderDto.getNickname() + "%");
		}
		if (StringUtils.isNotBlank(userOrderDto.getPhone())) {
			orderInfoCriteria.andEqualTo("phone", userOrderDto.getPhone());
		}
		if (userOrderDto.getType() != null) {
			orderInfoCriteria.andEqualTo("type", userOrderDto.getType());
		}
		if (userOrderDto.getSendType() != null) {
			orderInfoCriteria.andEqualTo("sendType", userOrderDto.getSendType());
		}
		if (userOrderDto.getInvoiceStatus() != null && userOrderDto.getInvoiceStatus() == 2) {
			orderInfoCriteria.andIsNotNull("invoiceId");
		}
		if (userOrderDto.getInvoiceStatus() != null && userOrderDto.getInvoiceStatus() == 1) {
			orderInfoCriteria.andIsNull("invoiceId");
		}
		if (StringUtils.isNotBlank(userOrderDto.getStartTime()) && StringUtils.isNotBlank(userOrderDto.getEndTime())) {
			orderInfoCriteria.andBetween("crtTime", userOrderDto.getStartTime(), userOrderDto.getEndTime());
		}
		if (StringUtils.isNotBlank(userOrderDto.getName())) {
			orderInfoCriteria.andLike("realName", "%" + userOrderDto.getName() + "%");
		}
		if (userOrderDto.getNewStatus() != null) {
			orderInfoCriteria.andEqualTo("status", userOrderDto.getNewStatus());
		}
		if (userOrderDto.getAuditStatus() != null && userOrderDto.getAuditStatus() == 1) {
			orderInfoCriteria.andEqualTo("auditStatus", userOrderDto.getAuditStatus());
		}
		if (userOrderDto.getAuditStatus() != null && userOrderDto.getAuditStatus() == 1) {
			orderInfoCriteria.andEqualTo("auditStatus", userOrderDto.getAuditStatus());
		}
		if (StringUtils.isNotBlank(userOrderDto.getKeywords())) {
			orderInfoCriteria.andLike("realName", "%" + userOrderDto.getKeywords() + "%")
					.orLike("realName", "%" + userOrderDto.getKeywords() + "%")
					.orLike("phone", "%" + userOrderDto.getKeywords() + "%")
					.orLike("orderNo", "%" + userOrderDto.getKeywords() + "%");
		}
//		example.orderBy("updTime").desc();
		example.orderBy("crtTime").desc();
		PageDataVO<OrderInfo> pageDataVO = PageDataVO.pageInfo(query, () -> mapper.selectByExample(example));
		if (pageDataVO != null && pageDataVO.getData() != null && pageDataVO.getData().size() > 0) {
			fillOrderItem(pageDataVO.getData());
			fillOrderEInvoice(pageDataVO.getData());
			return ObjectRestResponse.succ(pageDataVO);
		}
		return ObjectRestResponse.succ(pageDataVO);
	}


	/**
	 * 定制信息转订单
	 *
	 * @param customFormDto
	 * @return
	 */
	public ObjectRestResponse customerToOrderInfo(CustomFormDto customFormDto) {

		if (customFormDto == null || customFormDto.getCustomId() == null || customFormDto.getAuditStatus() == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		CustomForm customForm = customFormBiz.selectById(customFormDto.getCustomId());
		if (customForm == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.NOTEXIST_CODE, ResultCode.getMsg(ResultCode.NOTEXIST_CODE));
		}
		if (customFormDto.getAuditStatus() == 2) { //审核通过
			AppUserDTO appUserDTO = userFeign.userDetailById(customForm.getUserId()).getData();
			if (appUserDTO == null) {
				return ObjectRestResponse.createFailedResult(ResultCode.USER_NOTEXIST_CODE, ResultCode.getMsg(ResultCode.USER_NOTEXIST_CODE));
			}
			OrderInfo orderInfo = new OrderInfo();
			orderInfo.setRealName(appUserDTO.getRealname());
			orderInfo.setUserId(appUserDTO.getUserid());
			orderInfo.setPhone(appUserDTO.getUsername());
			orderInfo.setNickname(appUserDTO.getNickname());
			orderInfo.setHeadImg(appUserDTO.getHeadimgurl());
			orderInfo.setStatus(OrderStatusEnum.ORDER_UNPAID.getType());
			orderInfo.setCustomerId(customFormDto.getCustomId());
			orderInfo.setCustomerJson(JSONObject.toJSONString(customForm));
			orderInfo.setAmount(customFormDto.getAmount());
			orderInfo.setTotalAmount(customFormDto.getAmount());
			orderInfo.setNumber(1);
			orderInfo.setType(2);
			orderInfo.setSendType(1);
			orderInfo.setDataPrepare(0);
			generateOrder(orderInfo);
			customForm.setOrderId(Integer.parseInt(String.valueOf(orderInfo.getOrderId())));
			customForm.setOrderNo(orderInfo.getOrderNo());
			//设置定制信息为已转成订单
			customForm.setStatus(2);
			customForm.setAuditStatus(customFormDto.getAuditStatus());
			customForm.setAmount(customFormDto.getAmount());
			customFormBiz.updateSelectiveByIdRe(customForm);
			//添加通知信息
			noticeInfoBiz.addObj(DateUtil.date(customForm.getCrtTime()).toDateStr() + "提交的一个定制需求已被审核通过，生成订单" + orderInfo.getOrderNo() + "，待支付", NoticeTypeEnum.ORDER_INFO.getType(), appUserDTO.getUserid(), String.valueOf(orderInfo.getOrderId()));
			return ObjectRestResponse.succ();
		} else {
			customForm.setAuditStatus(customFormDto.getAuditStatus());
			customForm.setAuditReason(customFormDto.getAuditReason());
			customFormBiz.updateSelectiveByIdRe(customForm);
			return ObjectRestResponse.succ();
		}

	}


	/**
	 * 设置定制订单价格
	 *
	 * @param orderInfo
	 * @return
	 */
	public ObjectRestResponse setOrderPrice(OrderInfo orderInfo) {
		if (orderInfo == null || orderInfo.getOrderId() == null || orderInfo.getAmount() == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo old = selectById(orderInfo.getOrderId());
		if (old == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		old.setAmount(orderInfo.getAmount());
		old.setTotalAmount(orderInfo.getTotalAmount());
		updateSelectiveByIdRe(old);
		return ObjectRestResponse.succ();
	}

	/**
	 * 设置数据为已准备
	 * @param orderInfoDto
	 * @return
	 */
	public ObjectRestResponse setDataPrepare(OrderInfoDto orderInfoDto) {
		if (orderInfoDto == null || orderInfoDto.getOrderId() == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(orderInfoDto.getOrderId());
		if (orderInfo == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		orderInfo.setDataPrepare(1);
		updateSelectiveByIdRe(orderInfo);
		CustomForm customForm = customFormBiz.selectOne(new CustomForm() {{
			setOrderId(orderInfo.getOrderId().intValue());
		}});
		if (customForm != null) {
			customForm.setStatus(2);
			customFormBiz.updateSelectiveByIdRe(customForm);
		}
		return ObjectRestResponse.succ();
	}

	/**
	 * 定制订单添加订单商品
	 *
	 * @param orderInfoDto
	 * @return
	 */
	public ObjectRestResponse setOrderItem(OrderInfoDto orderInfoDto) {
		if (orderInfoDto == null || orderInfoDto.getOrderId() == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		List<ItemInfoDto> itemInfoDtoList = orderInfoDto.getItemInfoDtoList();
		if (itemInfoDtoList == null || itemInfoDtoList.size() <= 0) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(orderInfoDto.getOrderId());
		if (orderInfo == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		List<OrderItem> orderItems = orderItemBiz.selectList(new OrderItem(){{setOrderId(orderInfo.getOrderId());}});
		if (orderItems != null && orderItems.size() > 0) {
			orderItems.parallelStream().forEach(orderItem -> {
				orderItemBiz.delete(orderItem);
			});
		}
		List<OrderItem> orderItemList = new ArrayList<>();
		itemInfoDtoList.stream().forEach(itemInfoDto -> {
			switch (OrderTypeEnum.getByCode(itemInfoDto.getType())) {
				//标准数据
				case STANDARD_DATA:
					List<ImageDataVO> imageDataVOList = datacenterFeign.queryByIds(itemInfoDto.getId() + "").getData();
					if (imageDataVOList != null && imageDataVOList.size() > 0) {
						ImageDataVO imageDataVO = imageDataVOList.get(0);
						OrderItem orderItem = new OrderItem();
						orderItem.setType(itemInfoDto.getType());
						orderItem.setTotalAmount(imageDataVO.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber())));
						orderItem.setPrice(imageDataVO.getPrice());
						orderItem.setNumber(itemInfoDto.getNumber());
						orderItem.setDetailJson(JSONObject.toJSONString(imageDataVO));
						orderItem.setDetailId(Long.valueOf(imageDataVO.getId()));
						orderItemList.add(orderItem);
					}
					break;
				//行业应用
				case INDUSTRY_INFO:
					IndustryApplicationInfo industryApplicationInfo = industryApplicationInfoBiz.getOneById(itemInfoDto.getId().intValue()).getData();
					if (industryApplicationInfo != null) {
						OrderItem orderItem = new OrderItem();
						orderItem.setType(itemInfoDto.getType());
						orderItem.setTotalAmount(industryApplicationInfo.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber())));
						orderItem.setPrice(industryApplicationInfo.getPrice());
						orderItem.setNumber(itemInfoDto.getNumber());
						orderItem.setName(industryApplicationInfo.getTitle());
						orderItem.setItemPic(industryApplicationInfo.getCoverImg());
						orderItem.setDetailJson(JSONObject.toJSONString(industryApplicationInfo));
						orderItem.setDetailId(Long.valueOf(industryApplicationInfo.getId()));
						orderItemList.add(orderItem);
					}
					break;
				//影像图库
				case IMAGE_STORAGE:
					ImageInfoRelation imageInfoRelation = imageInfoRelationBiz.selectById(itemInfoDto.getId());
					if (imageInfoRelation != null) {
						ImageImgStorage imageImgStorage = imageImgStorageBiz.getDetail(imageInfoRelation.getRelationId()).getData();
						if (imageImgStorage != null) {
							List<ImageInfoRelation> imageInfoRelationList = new ArrayList<>();
							imageInfoRelationList.add(imageInfoRelation);
							imageImgStorage.setImageInfoRelationList(imageInfoRelationList);
							OrderItem orderItem = new OrderItem();
							orderItem.setType(itemInfoDto.getType());
							orderItem.setTotalAmount(imageInfoRelation.getPrice().multiply(new BigDecimal(itemInfoDto.getNumber())));
							orderItem.setPrice(imageInfoRelation.getPrice());
							orderItem.setNumber(itemInfoDto.getNumber());
							orderItem.setName(imageImgStorage.getName() + " (" + imageInfoRelation.getFileWidth()+ ")");
							orderItem.setItemPic(imageImgStorage.getCoverImg());
							orderItem.setDetailJson(JSONObject.toJSONString(imageImgStorage));
							orderItem.setDetailId(Long.valueOf(imageInfoRelation.getId()));
							orderItemList.add(orderItem);
						}
					}
					break;
			}
		});
		orderItemList.stream().forEach(orderItem -> {
			if (orderItem != null) {
				orderItem.setOrderId(orderInfoDto.getOrderId());
				orderItemBiz.insertSelective(orderItem);
			}
		});

		return ObjectRestResponse.succ();
	}

	/**
	 * 定制订单 添加发票和地址信息
	 *
	 * @param orderInfoDto
	 * @return
	 */
	public ObjectRestResponse setAddressOrInvoice(OrderInfoDto orderInfoDto) {
		if (orderInfoDto == null || orderInfoDto.getOrderId() == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		OrderInfo orderInfo = selectById(orderInfoDto.getOrderId());
		if (orderInfo == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		if (orderInfoDto.getUserAddressId() != null) {
			UserAddress userAddress = userAddressBiz.selectById(orderInfoDto.getUserAddressId());
			if (userAddress != null) {
				orderInfo.setReceiveName(userAddress.getConsigneeName());
				orderInfo.setReceivePhone(userAddress.getConsigneePhone());
				orderInfo.setReceiveAddress(userAddress.getProvince() + userAddress.getCity() + userAddress.getTown() + userAddress.getConsigneeAddress());
			}
		}
		//订单发票信息
		if (orderInfoDto.getUserInvoiceId() != null) {
			UserInvoice userInvoice = userInvoiceBiz.selectById(orderInfoDto.getUserInvoiceId());
			if (userInvoice != null) {
				OrderEInvoice orderEInvoice = new OrderEInvoice();
				orderEInvoice.setAmount(orderInfo.getAmount());
				orderEInvoice.setType(1);
				orderEInvoice.setOrderIds(orderInfo.getOrderId() + "");
				orderEInvoice.setTitleType(userInvoice.getTitleType());
				orderEInvoice.setTitleName(userInvoice.getTitleName());
				orderEInvoice.setTaxCode(userInvoice.getTaxCode());
				orderEInvoice.setContentRemark(orderInfoDto.getRemark());
				orderEInvoice.setContentAccount(userInvoice.getOpenBank() + "  " + userInvoice.getAccount());
				orderEInvoice.setContentContact(orderInfo.getReceiveAddress() + "  " + orderInfo.getReceiveName() + "  " + orderInfo.getReceivePhone());
				orderEInvoice.setStatus(2);
				//发票内容
				if (StringUtils.isNotBlank(orderInfoDto.getInvoiceContent())) {
					orderEInvoice.setContentTitle(orderInfoDto.getInvoiceContent());
				}
				orderEInvoiceBiz.addUpdate(orderEInvoice);
				orderInfo.setInvoiceStatus(1);
				orderInfo.setInvoiceId(orderEInvoice.getId());
			}
		}
		orderInfo.setSendType(orderInfoDto.getSendType());
		updateSelectiveByIdRe(orderInfo);
		return ObjectRestResponse.succ();
	}


	/**
	 * 根据类型查询商品信息
	 *
	 * @param type
	 * @param name
	 * @return
	 */
	public ObjectRestResponse getItemByType(Integer type, String name) {
		if (type == null) {
			return ObjectRestResponse.paramIsEmpty();
		}
		List<ItemInfoVo> itemInfoVoList = new ArrayList<>();
		switch (ItemTypeEnum.getByCode(type)) {
			case IMAGE_STORAGE:
				List<ImageImgStorage> imageImgStorageList = imageImgStorageBiz.getAll(null, name).getData();
				if (imageImgStorageList != null && imageImgStorageList.size() > 0) {
					imageImgStorageList.stream().forEach(imageImgStorage -> {
						List<ImageInfoRelation> imageInfoRelationList = imageImgStorage.getImageInfoRelationList();
						if (imageInfoRelationList != null && imageInfoRelationList.size() > 0) {
							imageInfoRelationList.stream().forEach(imageInfoRelation -> {
								ItemInfoVo itemInfoVo = new ItemInfoVo();
								itemInfoVo.setName(imageImgStorage.getName() + " (" + imageInfoRelation.getFileWidth()+ ")");
								itemInfoVo.setId(Long.parseLong(String.valueOf(imageInfoRelation.getId())));
								itemInfoVo.setType(ItemTypeEnum.IMAGE_STORAGE.getType());
								itemInfoVoList.add(itemInfoVo);
							});
						}
					});
				}
				break;
			case INDUSTRY_INFO:
				List<IndustryApplicationInfo> industryApplicationInfoList = industryApplicationInfoBiz.getAllData(name);
				if (industryApplicationInfoList != null && industryApplicationInfoList.size() > 0) {
					industryApplicationInfoList.stream().forEach(industryApplicationInfo -> {
						ItemInfoVo itemInfoVo = new ItemInfoVo();
						itemInfoVo.setName(industryApplicationInfo.getTitle());
						itemInfoVo.setId(Long.parseLong(String.valueOf(industryApplicationInfo.getId())));
						itemInfoVo.setType(ItemTypeEnum.INDUSTRY_INFO.getType());
						itemInfoVoList.add(itemInfoVo);
					});
				}
			case STANDARD_DATA:
				PageDataVO<ImageDataVO> pageDataVO = datacenterFeign.pageBykey(new ImagePageFeginDTO() {{
					setKeyword(name);
				}}).getData();
				if (pageDataVO != null && pageDataVO.getData() != null && pageDataVO.getData().size() > 0) {
					pageDataVO.getData().stream().forEach(imageDataVO -> {
						ItemInfoVo itemInfoVo = new ItemInfoVo();
						itemInfoVo.setName(imageDataVO.getName());
						itemInfoVo.setId(Long.parseLong(String.valueOf(imageDataVO.getId())));
						itemInfoVo.setType(ItemTypeEnum.STANDARD_DATA.getType());
						itemInfoVoList.add(itemInfoVo);
					});
				}
		}
		itemInfoVoList.removeIf(Objects::isNull);
		return ObjectRestResponse.succ(itemInfoVoList);
	}

	/**
	 * 订单支付
	 *
	 * @param orderPayVo
	 * @return
	 */
	public JSONObject pay(OrderPayVo orderPayVo) {
		if (orderPayVo == null || StringUtils.isBlank(orderPayVo.getOrderNo())) {
			return JsonResultUtil.createFailedResult(ResultCode.PARAM_ILLEGAL_CODE, ResultCode.getMsg(ResultCode.PARAM_ILLEGAL_CODE));
		}
		OrderInfo orderInfo = getOneByOrderNo(orderPayVo.getOrderNo());
		if (orderInfo == null) {
			return JsonResultUtil.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		orderInfo.setPayType(orderPayVo.getPayWay());
		orderInfo.setPayTime(new Date());
		orderInfo.setPayStatus(1);
		updateSelectiveByIdRe(orderInfo);
		orderPayVo.setAmount(orderInfo.getTotalAmount().multiply(new BigDecimal(100)).intValue());
		orderPayVo.setBuyerIp(StringUtils.defaultIfBlank(request.getHeader("userHost"), ClientUtil.getClientIp(request)));
		return thirdFeign.wx(orderPayVo);
	}

	/**
	 * 支付成功修改订单状态
	 *
	 * @param orderInfo
	 * @return
	 */
	public ObjectRestResponse updateOrderStatus(OrderInfo orderInfo) {
		log.info("回调修改订单状态：{}", orderInfo);
		if (orderInfo == null || StringUtils.isBlank(orderInfo.getOrderNo())) {
			return ObjectRestResponse.createFailedResult(ResultCode.PARAM_ILLEGAL_CODE, ResultCode.getMsg(ResultCode.PARAM_ILLEGAL_CODE));
		}
		OrderInfo old = getOneByOrderNo(orderInfo.getOrderNo());
		if (old == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		old.setStatus(OrderStatusEnum.ORDER_SEND.getType());
		old.setPayStatus(2);
		old.setPayTime(new Date());
		updateSelectiveByIdRe(old);
		//添加通知信息
		noticeInfoBiz.addObj("订单" + old.getOrderNo() + "已支付，待发货", NoticeTypeEnum.ORDER_INFO.getType(), old.getUserId(), String.valueOf(old.getOrderId()));
		return ObjectRestResponse.succ();
	}

	/**
	 * 对公账号支付设置
	 *
	 * @param orderInfo
	 * @return
	 */
	public ObjectRestResponse payByCompanyBank(OrderInfo orderInfo) {
		if (orderInfo == null || StringUtils.isBlank(orderInfo.getOrderNo())) {
			return ObjectRestResponse.createFailedResult(ResultCode.PARAM_ILLEGAL_CODE, ResultCode.getMsg(ResultCode.PARAM_ILLEGAL_CODE));
		}
		OrderInfo old = getOneByOrderNo(orderInfo.getOrderNo());
		if (old == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		old.setPayImg(orderInfo.getPayImg());
		old.setPayCode(orderInfo.getPayCode());
		old.setPayType(3);
		old.setPayTime(new Date());
		updateSelectiveByIdRe(old);
		return ObjectRestResponse.succ();
	}

	/**
	 * 后台确认对公支付成功
	 *
	 * @param orderInfo
	 * @return
	 */
	public ObjectRestResponse updatePayStatus(OrderInfo orderInfo) {
		if (orderInfo == null || StringUtils.isBlank(orderInfo.getOrderNo()) || orderInfo.getAuditStatus() == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.PARAM_ILLEGAL_CODE, ResultCode.getMsg(ResultCode.PARAM_ILLEGAL_CODE));
		}
		OrderInfo old = getOneByOrderNo(orderInfo.getOrderNo());
		if (old == null) {
			return ObjectRestResponse.createFailedResult(ResultCode.ORDER_NOT_EXIST, ResultCode.getMsg(ResultCode.ORDER_NOT_EXIST));
		}
		if (orderInfo.getAuditStatus() == 1) {
			old.setPayImg(orderInfo.getPayImg());
			old.setPayCode(orderInfo.getPayCode());
			old.setPayType(3);
			old.setPayTime(new Date());
			old.setAuditStatus(1);
			old.setPayStatus(2);
			old.setStatus(OrderStatusEnum.ORDER_SEND.getType());
			updateSelectiveByIdRe(old);
		} else {
			old.setAuditStatus(2);
			old.setPayStatus(1);
			old.setAuditReason(orderInfo.getAuditReason());
			updateSelectiveByIdRe(old);
		}
		return ObjectRestResponse.succ();
	}

}