package com.xxfc.platform.order.biz;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ObjectUtil;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.github.wxiaoqi.security.common.vo.PageDataVO;
import com.google.common.collect.Lists;
import com.xxfc.platform.order.bo.CompanyPerformanceBo;
import com.xxfc.platform.order.contant.enumerate.*;
import com.xxfc.platform.order.entity.*;
import com.xxfc.platform.order.mapper.OrderReceivedStatisticsMapper;
import com.xxfc.platform.order.pojo.dto.CompanyPerformanceFindDTO;
import com.xxfc.platform.order.pojo.dto.OrderReceivedStatisticsFindDTO;
import com.xxfc.platform.order.pojo.vo.OrderReceivedStatisticsVo;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.servlet.ServletOutputStream;
import java.io.IOException;
import java.math.BigDecimal;
import java.time.LocalDate;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 全部订单统计
 *
 * @author libin
 * @email 18178966185@163.com
 * @date 2019-11-08 18:03:42
 */
@Service
@Transactional(rollbackFor = Exception.class)
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
public class OrderReceivedStatisticsBiz extends BaseBiz<OrderReceivedStatisticsMapper, OrderReceivedStatistics> implements InitializingBean {
    private final OrderRentVehicleReceivedStatisticsBiz orderRentVehicleReceivedStatisticsBiz;
    private final OrderTourReceivedStatisticsBiz orderTourReceivedStatisticsBiz;
    private final OrderMemberReceivedStatisticsBiz orderMemberReceivedStatisticsBiz;

    private List<String> orderSigns;
    private List<String> orderMemberSigns;
    private List<String> orderRentVehicleSigns;
    private List<String> orderTourSigns;
    private Map<String, String> signAndDescMap;
    private DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern("yyyy-MM-dd");


    /**
     * 公司业绩（按日统计）
     * @param companyPerformanceFindDTO
     * @return
     */
    public PageDataVO<CompanyPerformanceBo> selectCompanyPerformanceWithDayPage(CompanyPerformanceFindDTO companyPerformanceFindDTO) {
         return PageDataVO.pageInfo(companyPerformanceFindDTO.getPage(), companyPerformanceFindDTO.getLimit(),
                () -> mapper.selectCompanyPerformanceWithDay(companyPerformanceFindDTO));
    }

    /**
     * 公司业绩（按月统计）
     * @param companyPerformanceFindDTO
     * @return
     */
    public PageDataVO<CompanyPerformanceBo> selectCompanyPerformanceWithMonthPage(CompanyPerformanceFindDTO companyPerformanceFindDTO) {
        return PageDataVO.pageInfo(companyPerformanceFindDTO.getPage(), companyPerformanceFindDTO.getLimit(),
                () -> mapper.selectCompanyPerformanceWithMonth(companyPerformanceFindDTO));
    }

    /**
     * 公司业绩（按周统计）
     * @param companyPerformanceFindDTO
     * @return
     */
    public PageDataVO<CompanyPerformanceBo> selectCompanyPerformanceWithWeekPage(CompanyPerformanceFindDTO companyPerformanceFindDTO) {
        return PageDataVO.pageInfo(companyPerformanceFindDTO.getPage(), companyPerformanceFindDTO.getLimit(),
                () -> mapper.selectCompanyPerformanceWithWeek(companyPerformanceFindDTO));
    }


    /**
     * 订单统计数据导出
     *
     * @param orderReceivedStatisticsFindDTO
     * @param outputStream
     */
    public void exportOrderReceivedStatisticsData(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO, ServletOutputStream outputStream) throws IOException {
        Map<String, List<OrderReceivedStatisticsVo>> orderReceivedStatisticsResult = getOrderReceivedStatisticsResult(orderReceivedStatisticsFindDTO);

        XSSFWorkbook workbook = new XSSFWorkbook();
        Sheet sheet = workbook.createSheet("订单统计");
        Row headerRow = sheet.createRow(0);
        //标题样式
        CellStyle headerCellStyle = createCellStyle(workbook, true);
        //创建标题
        createHeader(headerRow, headerCellStyle, orderReceivedStatisticsFindDTO.getStatisticalSigns());
        //创建数据样式
        CellStyle dataCellStyle = createCellStyle(workbook, false);
        createData(orderReceivedStatisticsResult, orderReceivedStatisticsFindDTO.getStatisticalWay(), orderReceivedStatisticsFindDTO.getStatisticalSigns(), sheet, 1, dataCellStyle);
        workbook.write(outputStream);
        workbook.close();
    }

    private void createData(Map<String, List<OrderReceivedStatisticsVo>> data, Integer statisticalWay, List<String> signs, Sheet sheet, int rowIndex, CellStyle cellStyle) {
        //自然排序
        Collections.sort(signs);
        int size = data.get(signs.get(0)).size();
        for (int i = 0; i < size; i++) {
            Row dataRow = sheet.createRow(rowIndex++);
            Cell firstCell = dataRow.createCell(0);
            firstCell.setCellStyle(cellStyle);
            for (int cellIndex = 1; cellIndex <= signs.size(); cellIndex++) {
                Cell dataCell = dataRow.createCell(cellIndex);
                dataCell.setCellStyle(cellStyle);
                String sign = signs.get(cellIndex - 1);
                List<OrderReceivedStatisticsVo> orderReceivedStatisticsVos = data.get(sign);
                if (CollectionUtils.isEmpty(orderReceivedStatisticsVos) || i > orderReceivedStatisticsVos.size()) {
                    continue;
                }
                OrderReceivedStatisticsVo orderReceivedStatisticsVo = orderReceivedStatisticsVos.get(i);
                if (sign.endsWith("q")) {
                    dataCell.setCellValue(String.valueOf(orderReceivedStatisticsVo.getOrderNum()));
                }
                if (sign.endsWith("a")) {
                    dataCell.setCellValue(String.valueOf(orderReceivedStatisticsVo.getOrderAmount()));
                }
                if (cellIndex == 1) {
                    if (ReceivedStatisticsEnum.DAY.getWayCode() == statisticalWay) {
                        firstCell.setCellValue(DateUtil.format(orderReceivedStatisticsVo.getDate(), "yyyy.MM.dd"));
                    }
                    if (ReceivedStatisticsEnum.WEEK.getWayCode() == statisticalWay) {
                        LocalDate localDate = LocalDate.of(Integer.valueOf(orderReceivedStatisticsVo.getYear()), Integer.valueOf(orderReceivedStatisticsVo.getMonth().replace(orderReceivedStatisticsVo.getYear(), "")), 1);
                        int dayOfYear = localDate.getDayOfYear();
                        int days = Integer.valueOf(orderReceivedStatisticsVo.getWeekOfYear().replace(orderReceivedStatisticsVo.getYear(), "")) * 7;
                        int betweenDays = days - dayOfYear;
                        LocalDate endLocalDate = localDate.withDayOfMonth(betweenDays);
                        LocalDate startLocalDate = endLocalDate.minusDays(6);
                        String startDateStr = startLocalDate.format(dateTimeFormatter);
                        String endDateStr = endLocalDate.format(dateTimeFormatter);
                        firstCell.setCellValue(String.format("%s~%s", startDateStr, endDateStr));
                    }
                    if (ReceivedStatisticsEnum.MONTH.getWayCode() == statisticalWay) {
                        StringBuilder sb = new StringBuilder(orderReceivedStatisticsVo.getMonth());
                        sb.insert(4, ".");
                        firstCell.setCellValue(sb.toString());
                    }
                }
            }
        }
    }

    private void createHeader(Row row, CellStyle cellStyle, List<String> signs) {
        //自然排序
        Collections.sort(signs);
        Cell firstCell = row.createCell(0);
        firstCell.setCellValue("时间");
        firstCell.setCellStyle(cellStyle);
        for (int i = 1; i <= signs.size(); i++) {
            Cell cell = row.createCell(i);
            cell.setCellValue(signAndDescMap.get(signs.get(i - 1)));
            cell.setCellStyle(cellStyle);
        }
    }

    private CellStyle createCellStyle(Workbook workbook, boolean bold) {
        CellStyle cellStyle = workbook.createCellStyle();
        Font font = createFont(workbook, bold);
        cellStyle.setFont(font);
        cellStyle.setVerticalAlignment(VerticalAlignment.CENTER);
        cellStyle.setAlignment(HorizontalAlignment.CENTER);
        return cellStyle;
    }

    private Font createFont(Workbook workbook, boolean bold) {
        Font font = workbook.createFont();
        font.setFontName("黑体");
        font.setBold(bold);
        font.setBold(true);
        return font;
    }

    /**
     * 获取订单统计数据
     *
     * @param orderReceivedStatisticsFindDTO
     * @return
     */
    public Map<String, List<OrderReceivedStatisticsVo>> getOrderReceivedStatisticsResult(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO) {
        Map<String, List<OrderReceivedStatisticsVo>> resultMap = new HashMap<>();
        List<String> statisticalSigns = orderReceivedStatisticsFindDTO.getStatisticalSigns();

        //订单统计
        Map<String, List<OrderReceivedStatisticsVo>> orderReceivedStatisticsMap = orderReceivedStatistics(orderReceivedStatisticsFindDTO, statisticalSigns);

        //租车订单统计
        Map<String, List<OrderReceivedStatisticsVo>> orderRentVehicleReceivedStatisticsMap = orderRentVehicleRceicedStatistics(orderReceivedStatisticsFindDTO, statisticalSigns);

        //旅游订单统计
        Map<String, List<OrderReceivedStatisticsVo>> orderTourReceivedStatisticsMap = orderTourReceivedStatistcs(orderReceivedStatisticsFindDTO, statisticalSigns);

        //会员订单统计
        Map<String, List<OrderReceivedStatisticsVo>> orderMemberReceivedStatisticsMap = orderMemberReceivedStatistics(orderReceivedStatisticsFindDTO, statisticalSigns);

        resultMap.putAll(orderReceivedStatisticsMap);
        resultMap.putAll(orderRentVehicleReceivedStatisticsMap);
        resultMap.putAll(orderTourReceivedStatisticsMap);
        resultMap.putAll(orderMemberReceivedStatisticsMap);
        return resultMap;
    }

    /**
     * 统计汇总
     *
     * @param orderMemberReceivedStatistics
     * @param orderTourReceivedStatistics
     * @param orderRentVehicleReceivedStatistics
     */
    public void orderReceivedStatistics(List<OrderMemberReceivedStatistics> orderMemberReceivedStatistics,
                                        List<OrderTourReceivedStatistics> orderTourReceivedStatistics,
                                        List<OrderRentVehicleReceivedStatistics> orderRentVehicleReceivedStatistics, Date date) {

        List<OrderReceivedStatistics> orderReceivedStatisticsList = new ArrayList<>();
        Map<String, List<OrderMemberReceivedStatistics>> memberMap = orderMemberReceivedStatistics.stream().collect(Collectors.groupingBy(OrderMemberReceivedStatistics::getStateGroup, Collectors.toList()));
        Map<String, List<OrderTourReceivedStatistics>> tourMap = orderTourReceivedStatistics.stream().collect(Collectors.groupingBy(OrderTourReceivedStatistics::getStateGroup, Collectors.toList()));
        Map<String, List<OrderRentVehicleReceivedStatistics>> rentVehicleMap = orderRentVehicleReceivedStatistics.stream().collect(Collectors.groupingBy(OrderRentVehicleReceivedStatistics::getStateGroup, Collectors.toList()));


        OrderReceivedStatistics orderReceivedStatistics = new OrderReceivedStatistics();
        Set<Map.Entry<String, List<OrderRentVehicleReceivedStatistics>>> rentVehicleSet = rentVehicleMap.entrySet();
        for (Map.Entry<String, List<OrderRentVehicleReceivedStatistics>> rentVehicleEntry : rentVehicleSet) {
            String stateGroup = rentVehicleEntry.getKey();
            List<OrderRentVehicleReceivedStatistics> rentVehicleEntryValue = rentVehicleEntry.getValue();
            OrderRentVehicleReceivedStatistics orderRentVehicle = rentVehicleEntryValue.get(0);
            OrderReceivedStatistics orderReceivedStatisticsClone = StatisticsStatusEnum.wrapStatisticsObject(date, stateGroup, Collections.EMPTY_MAP, ObjectUtil.clone(orderReceivedStatistics));
            orderReceivedStatisticsClone.setCompanyId(orderRentVehicle.getCompanyId());
            orderReceivedStatisticsClone.setCompanyName(orderRentVehicle.getCompanyName());

            OrderMemberReceivedStatistics orderMember = memberMap.get(stateGroup).get(0);
            OrderTourReceivedStatistics orderTour = tourMap==null?null:tourMap.get(stateGroup)==null?null:tourMap.get(stateGroup).get(0);



            orderReceivedStatisticsClone.setExtraAmount(orderRentVehicle.getExtraAmount());
            orderReceivedStatisticsClone.setLossSpecifiedAmount(orderRentVehicle.getLossSpecifiedAmount());

            orderReceivedStatisticsClone.setDepositAmount(orderRentVehicle.getDepositAmount());
            orderReceivedStatisticsClone.setDepositRefundAmount(orderRentVehicle.getDepositRefundAmount());
            orderReceivedStatisticsClone.setBreakRulesRegulationAmount(orderRentVehicle.getBreakRulesRegulationAmount());
            orderReceivedStatisticsClone.setNoDeductibleRefundAmount(orderRentVehicle.getNoDeductibleRefundAmount());
            orderReceivedStatisticsClone.setNoDeductibleAmount(orderRentVehicle.getNoDeductibleAmount());

            orderReceivedStatisticsClone.setTravelAmount(orderTour == null?BigDecimal.ZERO:orderTour.getTotalAmount());
            orderReceivedStatisticsClone.setMemberAmount(orderMember.getTotalAmount());
            orderReceivedStatisticsClone.setRentVehicleAmount(orderRentVehicle.getTotalAmount());

            BigDecimal totalAmount = orderRentVehicle.getTotalAmount().add(orderTour == null?BigDecimal.ZERO:orderTour.getTotalAmount()).add(orderMember.getTotalAmount());
            BigDecimal totalRefundAmount = orderRentVehicle.getOrderRefundAmount().add(orderTour==null?BigDecimal.ZERO:orderTour.getOrderRefundAmount()).add(orderMember.getOrderRefundAmount());
            BigDecimal lateFeeAmount = orderRentVehicle.getLateFeeAmount().add(orderTour==null?BigDecimal.ZERO:orderTour.getLateFeeAmount()).add(orderMember.getLateFeeAmount());
            Integer totalQuantity = orderRentVehicle.getTotalQuantity() + (orderTour == null?0:orderTour.getTotalQuantity()) + orderMember.getTotalQuantity();

            orderReceivedStatisticsClone.setLateFeeAmount(lateFeeAmount);
            orderReceivedStatisticsClone.setTotalAmount(totalAmount);
            orderReceivedStatisticsClone.setOrderRefundAmount(totalRefundAmount);
            orderReceivedStatisticsClone.setTotalQuantity(totalQuantity);
            orderReceivedStatisticsList.add(orderReceivedStatisticsClone);

        }

        insertReceivedStatisticsBatch(orderReceivedStatisticsList);

    }


    /**
     * 批量插入数据  mysql sql语句默认不能超过4M
     *
     * @param orderReceivedStatistics
     */
    public void insertReceivedStatisticsBatch(List<OrderReceivedStatistics> orderReceivedStatistics) {
        orderReceivedStatistics.sort(Comparator.comparing(OrderReceivedStatistics::getCompanyId));
        int orderSize = orderReceivedStatistics.size();
        int sqlAdq = orderSize / StatisticsStatusEnum.DEFAULT_SQL_SIZE;
        int sqlMod = orderSize % StatisticsStatusEnum.DEFAULT_SQL_SIZE;
        sqlAdq = sqlMod == 0 ? sqlAdq : sqlAdq + 1;
        for (int i = 0; i < sqlAdq; i++) {
            int fromIndex = StatisticsStatusEnum.DEFAULT_SQL_SIZE * i;
            int toIndex = StatisticsStatusEnum.DEFAULT_SQL_SIZE * (i + 1);
            toIndex = toIndex > orderSize ? orderSize : toIndex;
            List<OrderReceivedStatistics> orderRentVehicleReceivedStatisticsList = orderReceivedStatistics.subList(fromIndex, toIndex);
            mapper.insertList(orderRentVehicleReceivedStatisticsList);
        }
    }

    /**
     * 订单统计
     *
     * @param orderReceivedStatisticsFindDTO
     * @param statisticalSigns
     * @return
     */
    private Map<String, List<OrderReceivedStatisticsVo>> orderReceivedStatistics(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO, List<String> statisticalSigns) {
        if (CollectionUtil.containsAny(statisticalSigns, orderSigns)) {
            List<OrderReceivedStatistics> orderReceivedStatisticsList = selectOrderReceivedStatistics(orderReceivedStatisticsFindDTO);
            //日 周 月  的统计
            Map<Object, List<OrderReceivedStatisticsBase>> orderstatisticsMap = ReceivedStatisticsEnum.orderstatistics(orderReceivedStatisticsList, orderReceivedStatisticsFindDTO.getStatisticalWay());
            //统计项
            Map<String, List<OrderReceivedStatisticsVo>> orderReceivedStatisticsMap = OrderReceivedStatisticsSignEnum.orderReceivedStatistics(orderstatisticsMap, retainAll(orderSigns, statisticalSigns));
            return orderReceivedStatisticsMap.isEmpty() ? Collections.EMPTY_MAP : orderReceivedStatisticsMap;
        }
        return Collections.EMPTY_MAP;
    }

    /**
     * 租车统计
     *
     * @param orderReceivedStatisticsFindDTO
     * @param statisticalSigns
     * @return
     */
    private Map<String, List<OrderReceivedStatisticsVo>> orderRentVehicleRceicedStatistics(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO, List<String> statisticalSigns) {
        if (CollectionUtil.containsAny(statisticalSigns, orderRentVehicleSigns)) {
            List<OrderRentVehicleReceivedStatistics> orderRentVehicleReceivedStatisticsList = orderRentVehicleReceivedStatisticsBiz.selectOrderReceivedStatistics(orderReceivedStatisticsFindDTO);
            //日 周 月  的统计
            Map<Object, List<OrderReceivedStatisticsBase>> orderRentVehiclestatisticsMap = ReceivedStatisticsEnum.orderstatistics(orderRentVehicleReceivedStatisticsList, orderReceivedStatisticsFindDTO.getStatisticalWay());
            //统计项
            Map<String, List<OrderReceivedStatisticsVo>> orderRentVehicleReceivedStatisticsMap = OrderRentVehicleReceivedStatisticsSignEnum.orderMemberstatisticsMap(orderRentVehiclestatisticsMap, retainAll(orderRentVehicleSigns, statisticalSigns));
            return orderRentVehicleReceivedStatisticsMap.isEmpty() ? Collections.EMPTY_MAP : orderRentVehicleReceivedStatisticsMap;
        }
        return Collections.EMPTY_MAP;
    }

    /**
     * 旅游统计
     *
     * @param orderReceivedStatisticsFindDTO
     * @param statisticalSigns
     * @return
     */
    private Map<String, List<OrderReceivedStatisticsVo>> orderTourReceivedStatistcs(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO, List<String> statisticalSigns) {
        if (CollectionUtil.containsAny(statisticalSigns, orderTourSigns)) {
            List<OrderTourReceivedStatistics> orderTourReceivedStatisticsList = orderTourReceivedStatisticsBiz.selectOrderReceivedStatistics(orderReceivedStatisticsFindDTO);
            //日 周 月  的统计
            Map<Object, List<OrderReceivedStatisticsBase>> orderTourstatisticsMap = ReceivedStatisticsEnum.orderstatistics(orderTourReceivedStatisticsList, orderReceivedStatisticsFindDTO.getStatisticalWay());
            //统计项
            Map<String, List<OrderReceivedStatisticsVo>> orderTourReceivedStatisticsMap = OrderTourReceivedStatisticsSignEnum.orderMemberstatisticsMap(orderTourstatisticsMap, retainAll(orderTourSigns, statisticalSigns));
            return orderTourReceivedStatisticsMap.isEmpty() ? Collections.EMPTY_MAP : orderTourReceivedStatisticsMap;
        }
        return Collections.EMPTY_MAP;
    }

    /**
     * 会员统计
     *
     * @param orderReceivedStatisticsFindDTO
     * @param statisticalSigns
     * @return
     */
    private Map<String, List<OrderReceivedStatisticsVo>> orderMemberReceivedStatistics(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO, List<String> statisticalSigns) {
        if (CollectionUtil.containsAny(statisticalSigns, orderMemberSigns)) {
            List<OrderMemberReceivedStatistics> orderMemberReceivedStatisticsList = orderMemberReceivedStatisticsBiz.selectOrderReceivedStatistics(orderReceivedStatisticsFindDTO);
            //日 周 月  的统计
            Map<Object, List<OrderReceivedStatisticsBase>> orderMemberstatisticsMap = ReceivedStatisticsEnum.orderstatistics(orderMemberReceivedStatisticsList, orderReceivedStatisticsFindDTO.getStatisticalWay());
            //统计项
            Map<String, List<OrderReceivedStatisticsVo>> orderMemberReceivedStatisticsMap = OrderMemberReceivedStatisticsSignEnum.orderMemberstatisticsMap(orderMemberstatisticsMap, retainAll(orderMemberSigns, statisticalSigns));
            return orderMemberReceivedStatisticsMap.isEmpty() ? Collections.EMPTY_MAP : orderMemberReceivedStatisticsMap;
        }
        return Collections.EMPTY_MAP;
    }

    private List<OrderReceivedStatistics> selectOrderReceivedStatistics(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO) {
        return mapper.selectOrderReceivedStatisticsList(orderReceivedStatisticsFindDTO);
    }

    /**
     * 取两个集合的交集
     *
     * @param source
     * @param target
     * @return
     */
    private List<String> retainAll(List<String> source, List<String> target) {
        ArrayList<String> sourceBak = Lists.newArrayList(source);
        sourceBak.retainAll(target);
        return sourceBak;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        signAndDescMap = new HashMap<>();
        orderSigns = EnumSet.allOf(OrderReceivedStatisticsSignEnum.class).stream().map(OrderReceivedStatisticsSignEnum::getSign).collect(Collectors.toList());
        Map<String, String> orderSignAndDescMap = EnumSet.allOf(OrderReceivedStatisticsSignEnum.class).stream().collect(Collectors.toMap(OrderReceivedStatisticsSignEnum::getSign, OrderReceivedStatisticsSignEnum::getDesc));
        signAndDescMap.putAll(orderSignAndDescMap);
        orderMemberSigns = EnumSet.allOf(OrderMemberReceivedStatisticsSignEnum.class).stream().map(OrderMemberReceivedStatisticsSignEnum::getSign).collect(Collectors.toList());
        Map<String, String> orderMemberSignAndDescMap = EnumSet.allOf(OrderMemberReceivedStatisticsSignEnum.class).stream().collect(Collectors.toMap(OrderMemberReceivedStatisticsSignEnum::getSign, OrderMemberReceivedStatisticsSignEnum::getDesc));
        signAndDescMap.putAll(orderMemberSignAndDescMap);
        orderRentVehicleSigns = EnumSet.allOf(OrderRentVehicleReceivedStatisticsSignEnum.class).stream().map(OrderRentVehicleReceivedStatisticsSignEnum::getSign).collect(Collectors.toList());
        Map<String, String> orderRentVehicleSignAndDescMap = EnumSet.allOf(OrderRentVehicleReceivedStatisticsSignEnum.class).stream().collect(Collectors.toMap(OrderRentVehicleReceivedStatisticsSignEnum::getSign, OrderRentVehicleReceivedStatisticsSignEnum::getDesc));
        signAndDescMap.putAll(orderRentVehicleSignAndDescMap);
        orderTourSigns = EnumSet.allOf(OrderTourReceivedStatisticsSignEnum.class).stream().map(OrderTourReceivedStatisticsSignEnum::getSign).collect(Collectors.toList());
        Map<String, String> orderTourSignAndDescMap = EnumSet.allOf(OrderTourReceivedStatisticsSignEnum.class).stream().collect(Collectors.toMap(OrderTourReceivedStatisticsSignEnum::getSign, OrderTourReceivedStatisticsSignEnum::getDesc));
        signAndDescMap.putAll(orderTourSignAndDescMap);
    }


}