package com.xxfc.platform.order.biz;

import cn.hutool.core.date.DateUtil;
import com.alibaba.fastjson.JSON;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.github.wxiaoqi.security.common.util.CollectorsUtil;
import com.xxfc.platform.order.bo.LargeScreenDisplayConstantDataBo;
import com.xxfc.platform.order.contant.enumerate.OrderTypeEnum;
import com.xxfc.platform.order.entity.OrderProfileDisplay;
import com.xxfc.platform.order.mapper.OrderProfileDisplayMapper;
import com.xxfc.platform.order.pojo.dto.BaseOrderDTO;
import com.xxfc.platform.order.pojo.vo.*;
import com.xxfc.platform.universal.entity.Dictionary;
import com.xxfc.platform.universal.feign.ThirdFeign;
import com.xxfc.platform.vehicle.entity.Area;
import com.xxfc.platform.vehicle.feign.VehicleFeign;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author libin
 * @version 1.0
 * @description
 * @data 2019/12/24 17:06
 */
@Slf4j
@Transactional(rollbackFor = Exception.class)
@Service
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
public class OrderProfileDisplayBiz extends BaseBiz<OrderProfileDisplayMapper, OrderProfileDisplay> {
    /**
     * 字典查询type
     */
    @Value("large.screen.type:LARGE_SCREEN_DISPLAY")
    private String largeScreenType;
    /**
     * 字典查询code
     */
    @Value("large.screen.code:LARGE_SCREEN_DISPLAY_CONSTANT")
    private String largeScreenCode;

    private final ThirdFeign thirdFeign;
    private final VehicleFeign vehicleFeign;
    private final BaseOrderBiz baseOrderBiz;
    private final ThreadPoolTaskExecutor executor;
    private static final Integer WX_PAY_STATE = 1;
    private static final Integer ALI_PAY_STATE = 2;
    private static final Integer APP_PAY_STATE = 1;
    private static final Integer APPLET_PAY_STATE = 2;
    private static final Integer WECHAT_OFFICIAL_ACCOUNT_PAY_STATE = 3;
    private static final Integer IOS_PAY_STATE = 4;

    /**
     * @param startDate 开始时间
     * @param endDate   结束时间
     * @return
     */
    public LargeScreenDisplayDataVo findLargeScreenDisplayDataByDate(Date startDate, Date endDate) {
        LargeScreenDisplayDataVo largeScreenDisplayDataVo = new LargeScreenDisplayDataVo();
        //创建订单常量数据对象
        LargeScreenDisplayConstantDataBo largeScreenDisplayConstantDataBo = new LargeScreenDisplayConstantDataBo();
        //创建订单数据统计对象
        OrderProfileDispayVo orderProfileDispayVo = new OrderProfileDispayVo();

        //1.查询基础常量数据
        Dictionary largeScreenConstantDictionary = thirdFeign.findDictionaryByTypeAndCode(largeScreenType, largeScreenCode);
        if (Objects.nonNull(largeScreenConstantDictionary) && StringUtils.hasText(largeScreenConstantDictionary.getDetail())) {
            largeScreenDisplayConstantDataBo = JSON.parseObject(largeScreenConstantDictionary.getDetail(), LargeScreenDisplayConstantDataBo.class);
        }

        //2.1查询订单数据
        List<BaseOrderDTO> baseOrders = baseOrderBiz.findOrdersByDate(startDate, endDate);
        //2.2查询固定数据
        List<OrderProfileDisplay> orderProfileDisplays = findOrderProfileDisplayDataByDate(startDate, endDate);

        //包装成基础订单
        List<BaseOrderDTO> wrapBaseOrders = wrapToBaseOrder(orderProfileDisplays);
        wrapBaseOrders.addAll(baseOrders);

        Supplier<Stream<BaseOrderDTO>> orderSupplier = () -> wrapBaseOrders.stream();
        CountDownLatch latch = new CountDownLatch(3);
        //3. 十大运营中心订单数据
        executor.execute(() -> {
            try {
                analyticStatisticsCenterOrder(orderSupplier, orderProfileDispayVo);
            } catch (Exception ex) {
                log.error("十大运营中心订单统计失败【{}】", ex);
            } finally {
                latch.countDown();
            }
        });
        //4.总的订单数据
        executor.execute(() -> {
            try {
                analyticStatisticsOrderProfiles(orderSupplier, orderProfileDispayVo);
            } catch (Exception ex) {
                log.error("总订单数据解析失败【{}】", ex);
            } finally {
                latch.countDown();
            }
        });
        //5.支付方式|支付终端数据处理
        executor.execute(() -> {
            try {
                analyticStatisticsOrderPayWayProfiles(orderSupplier, orderProfileDispayVo);
            } catch (Exception ex) {
                log.error("支付方式|支付终端数据处理失败【{}】", ex);
            } finally {
                latch.countDown();
            }
        });
        try {
            latch.await();
        } catch (InterruptedException e) {
            log.error("统计出错：【{}】", e);
        }
        //6.基础金额 + 真实金额  基础订单量+真实订单量
        long totalOrderNum = orderProfileDispayVo.getOrderNum() + largeScreenDisplayConstantDataBo.getBaseOrderNum();
        BigDecimal totalOrderAmount = orderProfileDispayVo.getOrderAmount().add(largeScreenDisplayConstantDataBo.getBaseOrderAmount());

        orderProfileDispayVo.setOrderNum(totalOrderNum);
        orderProfileDispayVo.setOrderAmount(totalOrderAmount);
        largeScreenDisplayDataVo.setLargeScreenDisplayConstantData(largeScreenDisplayConstantDataBo);
        largeScreenDisplayDataVo.setOrderProfileDispayData(orderProfileDispayVo);
        return largeScreenDisplayDataVo;
    }

    /**
     * 支付方式|终端的统计
     *
     * @param orderSuppiler
     * @param orderProfileDispayVo
     * @return
     */
    private OrderProfileDispayVo analyticStatisticsOrderPayWayProfiles(Supplier<Stream<BaseOrderDTO>> orderSuppiler,
                                                                       OrderProfileDispayVo orderProfileDispayVo) {
        OrderPayProfileDispalyVo orderPayProfileDispalyVo = new OrderPayProfileDispalyVo();
        //1.支付方式统计
        wrapToPayProfileWithPayWay(orderSuppiler, orderPayProfileDispalyVo);
        //2.支付终端统计
        wrapToPayProfileWithPayTerminal(orderSuppiler, orderPayProfileDispalyVo);

        orderProfileDispayVo.setOrderPayProfileDisplay(orderPayProfileDispalyVo);
        return orderProfileDispayVo;
    }

    /**
     * 支付方式统计
     *
     * @param orderSupplier
     * @param orderPayProfileDispalyVo
     * @return
     */
    private OrderPayProfileDispalyVo wrapToPayProfileWithPayWay(Supplier<Stream<BaseOrderDTO>> orderSupplier, OrderPayProfileDispalyVo orderPayProfileDispalyVo) {
        Map<Integer, BigDecimal> payWayMap = orderSupplier.get().collect(Collectors.groupingBy(BaseOrderDTO::getPayWay, CollectorsUtil.summingBigDecimal(BaseOrderDTO::getRealAmount)));
        boolean isNull = payWayMap == null;
        BigDecimal wxPayAmount = isNull ? BigDecimal.ZERO : payWayMap.get(WX_PAY_STATE) == null ? BigDecimal.ZERO : payWayMap.get(WX_PAY_STATE);
        BigDecimal aliPayAmount = isNull ? BigDecimal.ZERO : payWayMap.get(ALI_PAY_STATE) == null ? BigDecimal.ZERO : payWayMap.get(ALI_PAY_STATE);
        orderPayProfileDispalyVo.setWxPayAmount(wxPayAmount);
        orderPayProfileDispalyVo.setAliPayAmount(aliPayAmount);
        return orderPayProfileDispalyVo;
    }

    /**
     * 支付终端统计
     *
     * @param orderSupplier
     * @param orderPayProfileDispalyVo
     * @return
     */
    private OrderPayProfileDispalyVo wrapToPayProfileWithPayTerminal(Supplier<Stream<BaseOrderDTO>> orderSupplier, OrderPayProfileDispalyVo orderPayProfileDispalyVo) {
        Map<Integer, BigDecimal> payTerminalMap = orderSupplier.get().collect(Collectors.groupingBy(BaseOrderDTO::getPayOrigin, CollectorsUtil.summingBigDecimal(BaseOrderDTO::getRealAmount)));
        boolean isNull = payTerminalMap == null;
        BigDecimal appPayAmount = isNull ? BigDecimal.ZERO : payTerminalMap.get(APP_PAY_STATE) == null ? BigDecimal.ZERO : payTerminalMap.get(APP_PAY_STATE);
        BigDecimal appletPayAmount = isNull ? BigDecimal.ZERO : payTerminalMap.get(APPLET_PAY_STATE) == null ? BigDecimal.ZERO : payTerminalMap.get(APPLET_PAY_STATE);
        BigDecimal weChatOfficialAccountPayAmount = isNull ? BigDecimal.ZERO : payTerminalMap.get(WECHAT_OFFICIAL_ACCOUNT_PAY_STATE) == null ? BigDecimal.ZERO : payTerminalMap.get(WECHAT_OFFICIAL_ACCOUNT_PAY_STATE);
        BigDecimal iosPayAmount = isNull ? BigDecimal.ZERO : payTerminalMap.get(IOS_PAY_STATE) == null ? BigDecimal.ZERO : payTerminalMap.get(IOS_PAY_STATE);

        orderPayProfileDispalyVo.setAppletPayAmount(appPayAmount);
        orderPayProfileDispalyVo.setAppletPayAmount(appletPayAmount);
        orderPayProfileDispalyVo.setWeChatOfficialAccountPayAmount(weChatOfficialAccountPayAmount);
        orderPayProfileDispalyVo.setIosPayAmount(iosPayAmount);
        return orderPayProfileDispalyVo;
    }

    /**
     * 订单统计解析
     *
     * @param
     * @param orderProfileDispayVo
     * @return
     */
    private OrderProfileDispayVo analyticStatisticsOrderProfiles(Supplier<Stream<BaseOrderDTO>> orderSuppiler,
                                                                 OrderProfileDispayVo orderProfileDispayVo) {
        //1.真实数据按时间分组
        Map<Date, BigDecimal> orderAmountMap = orderSuppiler.get().collect(Collectors.groupingBy(BaseOrderDTO::getPayDate, CollectorsUtil.summingBigDecimal(BaseOrderDTO::getRealAmount)));
        //2.某个时间段的订单统计
        //2.1 订单金额
        List<OrderProfileVo> orderProfileVos = createOrderProfiles(orderAmountMap);
        //2.2 订单量
        wrapBaseDateToOrderProfileDispay(orderSuppiler, orderProfileDispayVo);
        //3.今日金额
        Date date = DateUtil.beginOfDay(new Date()).toJdkDate();
        BigDecimal todayOrderAmount = orderAmountMap == null ? BigDecimal.ZERO : orderAmountMap.get(date) == null ? BigDecimal.ZERO : orderAmountMap.get(date);
        orderProfileDispayVo.setTodayOrderAmount(todayOrderAmount);
        orderProfileDispayVo.setOrderProfiles(orderProfileVos);
        return orderProfileDispayVo;
    }

    /**
     * 包装基础数据  订单金额 与订单量
     *
     * @param orderSuppiler
     * @param orderProfileDispayVo
     * @return
     */
    private OrderProfileDispayVo wrapBaseDateToOrderProfileDispay(Supplier<Stream<BaseOrderDTO>> orderSuppiler, OrderProfileDispayVo orderProfileDispayVo) {
        Map<Date, Long> orderNumMap = orderSuppiler.get().collect(Collectors.groupingBy(BaseOrderDTO::getPayDate, Collectors.counting()));
        if (orderNumMap == null || orderNumMap.isEmpty()) {
            return orderProfileDispayVo;
        }
        Date date = DateUtil.beginOfDay(new Date()).toJdkDate();
        //今日订单量
        long todayOrderNum = orderNumMap.get(date) == null ? 0 : orderNumMap.get(date);
        Set<Map.Entry<Date, Long>> orderNumEntrySet = orderNumMap.entrySet();
        Iterator<Map.Entry<Date, Long>> orderNumIterator = orderNumEntrySet.iterator();
        //总订单量累计
        long totalOrderNum = 0;
        while (orderNumIterator.hasNext()) {
            Map.Entry<Date, Long> orderNumMapEntry = orderNumIterator.next();
            totalOrderNum = +orderNumMapEntry.getValue();
        }
        orderProfileDispayVo.setOrderNum(totalOrderNum);
        orderProfileDispayVo.setTodayOrderNum(todayOrderNum);
        return orderProfileDispayVo;
    }

    /**
     * 组装对象
     *
     * @param orderAmountMap
     * @return
     */
    private List<OrderProfileVo> createOrderProfiles(Map<Date, BigDecimal> orderAmountMap) {
        List<OrderProfileVo> orderProfileVos = new ArrayList<>();
        if (orderAmountMap == null || orderAmountMap.isEmpty()) {
            return orderProfileVos;
        }
        Set<Map.Entry<Date, BigDecimal>> orderAmountEntrySet = orderAmountMap.entrySet();
        Iterator<Map.Entry<Date, BigDecimal>> orderAmountIterator = orderAmountEntrySet.iterator();
        OrderProfileVo orderProfileVo;
        while (orderAmountIterator.hasNext()) {
            Map.Entry<Date, BigDecimal> orderAmountMapEntry = orderAmountIterator.next();
            orderProfileVo = new OrderProfileVo();
            orderProfileVo.setOrderDate(orderAmountMapEntry.getKey());
            orderProfileVo.setOrderAmount(orderAmountMapEntry.getValue());
            orderProfileVos.add(orderProfileVo);
        }
        return orderProfileVos;
    }

    /**
     * 运营中心订单统计
     *
     * @param orderSuppiler
     * @param orderProfileDispayVo
     * @return
     */
    private OrderProfileDispayVo analyticStatisticsCenterOrder(Supplier<Stream<BaseOrderDTO>> orderSuppiler,
                                                               OrderProfileDispayVo orderProfileDispayVo) {
        //筛选出租车订单
        Map<Integer, BigDecimal> areaAmountMap = orderSuppiler.get().filter(x -> Objects.equals(x.getType(), OrderTypeEnum.RENT_VEHICLE.getCode()))
                .collect(Collectors.groupingBy(BaseOrderDTO::getAreaId, CollectorsUtil.summingBigDecimal(BaseOrderDTO::getRealAmount)));

        List<CenterOrderProfileDisplayVo> centerOrderProfileDisplayVos = createCenterOrderProfileDisplay(areaAmountMap);
        orderProfileDispayVo.setCenterOrderProfileDisplays(centerOrderProfileDisplayVos);
        return orderProfileDispayVo;
    }

    /**
     * 区域统计处理
     *
     * @param areaAmountMap
     * @return
     */
    private List<CenterOrderProfileDisplayVo> createCenterOrderProfileDisplay(Map<Integer, BigDecimal> areaAmountMap) {
        List<CenterOrderProfileDisplayVo> centerOrderProfileDisplayVos = new ArrayList<>();
        //查询全部区域
        List<Area> areas = vehicleFeign.findAllAreas();
        if (CollectionUtils.isEmpty(areas)) {
            return centerOrderProfileDisplayVos;
        }
        CenterOrderProfileDisplayVo centerOrderProfileDisplayVo;
        //区域统计组装
        for (Area area : areas) {
            centerOrderProfileDisplayVo = new CenterOrderProfileDisplayVo();
            Integer areaId = area.getId();
            BigDecimal areaAmount = areaAmountMap == null ? BigDecimal.ZERO : areaAmountMap.get(areaId) == null ? BigDecimal.ZERO : areaAmountMap.get(areaId);
            centerOrderProfileDisplayVo.setAreaId(areaId);
            centerOrderProfileDisplayVo.setAreaName(area.getName());
            centerOrderProfileDisplayVo.setOrderAmount(areaAmount);
            centerOrderProfileDisplayVos.add(centerOrderProfileDisplayVo);
        }
        return centerOrderProfileDisplayVos;
    }

    /**
     * 包装成baseOrder
     *
     * @param orderProfileDisplays
     * @return
     */
    private List<BaseOrderDTO> wrapToBaseOrder(List<OrderProfileDisplay> orderProfileDisplays) {
        List<BaseOrderDTO> baseOrderDTOS = orderProfileDisplays.stream().map(x -> {
            BaseOrderDTO baseOrderDTO = new BaseOrderDTO();
            baseOrderDTO.setPayOrigin(x.getPayTerminal());
            baseOrderDTO.setPayWay(x.getPayWay());
            baseOrderDTO.setRealAmount(x.getOrderAmount());
            baseOrderDTO.setOrderNum(x.getOrderNum());
            baseOrderDTO.setAreaId(x.getAreaId());
            return baseOrderDTO;
        }).collect(Collectors.toList());
        return CollectionUtils.isEmpty(baseOrderDTOS) ? new ArrayList<>() : baseOrderDTOS;
    }


    /**
     * 根据开始时间与结束时间查询
     *
     * @param startDate
     * @param endDate
     * @return
     */
    public List<OrderProfileDisplay> findOrderProfileDisplayDataByDate(Date startDate, Date endDate) {
        List<OrderProfileDisplay> orderProfileDisplays = mapper.findOrderProfileDisplayDatabyDate(startDate, endDate);
        return CollectionUtils.isEmpty(orderProfileDisplays) ? Collections.emptyList() : orderProfileDisplays;
    }
}
