package com.xxfc.platform.vehicle.biz;

import cn.hutool.core.bean.BeanUtil;
import com.ace.cache.annotation.Cache;
import com.ace.cache.annotation.CacheClear;
import com.github.pagehelper.PageHelper;
import com.github.pagehelper.PageInfo;
import com.github.wxiaoqi.security.admin.feign.dto.UserDTO;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.google.common.collect.Lists;
import com.xxfc.platform.vehicle.common.RestResponse;
import com.xxfc.platform.vehicle.constant.RedisKey;
import com.xxfc.platform.vehicle.entity.BranchCompany;
import com.xxfc.platform.vehicle.entity.BranchCompanyStockInfo;
import com.xxfc.platform.vehicle.entity.SysRegion;
import com.xxfc.platform.vehicle.mapper.BranchCompanyMapper;
import com.xxfc.platform.vehicle.util.excel.ExcelImport;
import com.xxfc.platform.vehicle.pojo.BranchCompanyVo;
import com.xxfc.platform.vehicle.pojo.CompanyDetail;
import com.xxfc.platform.vehicle.pojo.CompanySearchDTO;
import com.github.wxiaoqi.security.common.vo.PageDataVO;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;
import tk.mybatis.mapper.entity.Example;

import javax.servlet.http.HttpServletRequest;
import java.io.File;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.github.wxiaoqi.security.auth.common.constatns.CommonConstants.DATA_ALL_FALSE;
import static com.xxfc.platform.vehicle.constant.DbColumnConstant.*;
import static com.xxfc.platform.vehicle.constant.RedisKey.BRANCH_COMPANY_CACHE_DATAZONE;

@Service
@Slf4j
public class BranchCompanyBiz extends BaseBiz<BranchCompanyMapper, BranchCompany> {

    public static final DateTimeFormatter DEFAULT_DATE_TIME_FORMATTER = DateTimeFormat.forPattern("yyyy-MM-dd");
    public static final DateTimeFormatter YEARMONTH_DATE_TIME_FORMATTER = DateTimeFormat.forPattern("yyyy-MM");

    @Autowired
    private RedisTemplate customRedisTemplate;

    @Autowired
    private SysRegionBiz sysRegionBiz;

    @Value("${branchCompanyPic.baseUploadPath}")
    private String baseUploadPath ;

    /**
     * 按主键获取公司
     * @param id
     * @return
     */
    public BranchCompany getById(Integer id) {
        return mapper.selectByPrimaryKey(id);
    }

    public List<BranchCompany> selectByZoneId(Integer zoneId) {
        if(zoneId == null) {
            return null;
        }
        return mapper.selectByZoneId(zoneId);
    }

    /**
     * 获取公司信息
     * @param id
     * @return
     */
    @Cache(key= RedisKey.BRANCH_COMPANY_CACHE)
    public CompanyDetail getDetailById(Integer id) {
        BranchCompany branchCompany = this.getById(id);
        CompanyDetail detail = null;
        if(branchCompany != null) {
            List<SysRegion> sysRegions = sysRegionBiz.getRegionsByCodes(new ArrayList<Long>(){{
                add(Long.valueOf(branchCompany.getAddrProvince()));
                add(Long.valueOf(branchCompany.getAddrCity()));
                add(Long.valueOf(branchCompany.getAddrTown()));
            }});
            detail = BeanUtil.toBean(branchCompany, CompanyDetail.class);
            detail.setSysRegions(sysRegions);
        }
        return detail;
    }

    public Boolean exits(Integer id){
        BranchCompany param = new BranchCompany();
        param.setId(id);
        return mapper.selectCount(param)>0;
    }

    /**
     * 转换为vo类
     * @param branchCompanies
     * @returns
     */
    private List<BranchCompanyVo> branchCompany2VO(List<BranchCompany> branchCompanies){
        List<BranchCompanyVo> branchCompanyVoList = Lists.newArrayList();
        if(CollectionUtils.isNotEmpty(branchCompanies)){
            for(BranchCompany branchCompany:branchCompanies){
                BranchCompanyVo branchCompanyVo = new BranchCompanyVo();
                BeanUtils.copyProperties(branchCompany,branchCompanyVo);
                branchCompanyVoList.add(branchCompanyVo);
            }
        }
        return  branchCompanyVoList;
    }

    public PageDataVO<BranchCompany> getAll(Integer page, Integer limit, Integer addrProvince, Integer addrCity,
                                            Integer addrTown, UserDTO userDTO){
        Example example = new Example(BranchCompany.class);
        Example.Criteria criteria = example.createCriteria();
        if(addrProvince!=null) {
            criteria.andCondition(" addr_province = '" + addrProvince + "'");
        }
        if(addrCity!=null) {
            criteria.andCondition(" addr_city = '" + addrCity + "'");
        }
        if(addrTown!=null) {
            criteria.andCondition(" addr_town = '" + addrTown + "'");
        }
        if(userDTO != null && DATA_ALL_FALSE.equals(userDTO.getDataAll())) {
            if(StringUtils.isNotBlank(userDTO.getDataZone())) {
                criteria.andIn(COMPANY_ZONE_ID, userDTO.dataZone2List());
            }
            if(StringUtils.isNotBlank(userDTO.getDataCompany())) {
                criteria.andIn(ID, userDTO.dataCompany2List());
            }
        }
        example.setOrderByClause("`id` asc");
        PageHelper.startPage(page,limit);
        PageInfo<BranchCompany> branchCompanyPageInfo = new PageInfo<>(mapper.selectByExample(example));
        return PageDataVO.pageInfo(branchCompanyPageInfo);
    }

    public PageDataVO<BranchCompany> search(CompanySearchDTO vo) {
        PageHelper.startPage(vo.getPage(), vo.getLimit());
        PageInfo<BranchCompany> branchCompanyPageInfo = new PageInfo<>(mapper.search(vo.getLon(), vo.getLat(), vo.getAddrCity()));
        return PageDataVO.pageInfo(branchCompanyPageInfo);
    }



    @Cache(key= RedisKey.BRANCH_COMPANY_CACHE_ALL)
    public List<BranchCompany> getAll(){
        return mapper.selectAll();
    }

    /**
     * 增加公司，返回id
     * @param branchCompanyVo
     * @return
     */
    @CacheClear(pre= RedisKey.BRANCH_COMPANY_CACHE)
    public Integer add(BranchCompanyVo branchCompanyVo){
        BranchCompany branchCompany = new BranchCompany();
        BeanUtils.copyProperties(branchCompanyVo,branchCompany);
        mapper.insertSelective(branchCompany);
        return branchCompany.getId();
    }

    @CacheClear(pre= RedisKey.BRANCH_COMPANY_CACHE)
    public void del(Integer id){
        mapper.deleteByPrimaryKey(id);
    }

    /**
     * 更新公司信息
     * @param branchCompany
     * @return
     */
    @CacheClear(pre = RedisKey.BRANCH_COMPANY_CACHE)
    public Integer update(BranchCompany branchCompany){
        return mapper.updateByPrimaryKeySelective(branchCompany);
    }

    /**
     * 写入上传文件，返回相对路径
     * @param file
     * @return
     */
    public RestResponse<String> uploadCompanyPic(MultipartFile file) throws Exception{
        //创建本日存放目录
        DateTime now = DateTime.now();
        String dirPathToday =  File.separator + now.toString(DEFAULT_DATE_TIME_FORMATTER);
        String redisNoKey = RedisKey.UPLOAD_FILE_NO_PREFIX + now.toString(DEFAULT_DATE_TIME_FORMATTER);
        Long no = customRedisTemplate.opsForValue().increment(redisNoKey);
        if(no.equals(1l)){
            customRedisTemplate.expire(redisNoKey,1, TimeUnit.DAYS);
        }
        String fileName = file.getOriginalFilename();
        String realFileRelPath = dirPathToday + File.separator + no + fileName.substring(fileName.lastIndexOf("."));
        //文件存放路径
        String filePath = baseUploadPath + realFileRelPath;
        //将文件写入指定位置
        FileUtils.copyInputStreamToFile(file.getInputStream(), new File(filePath));
        return RestResponse.suc(realFileRelPath);
    }

    @Cache(key=BRANCH_COMPANY_CACHE_DATAZONE)
    public List<BranchCompany> dataCompany(String dataZone) {
        if(StringUtils.isBlank(dataZone)) {
            return new ArrayList<BranchCompany>();
        }
        List<Integer> zoneIds = Arrays.asList(dataZone.split(",")).parallelStream().map(s -> Integer.valueOf(s)).collect(Collectors.toList());
        Example example = new Example(BranchCompany.class);
        example.createCriteria().andIn("zoneId", zoneIds);
        List<BranchCompany> zoneCompanys = selectByExample(example);
        return zoneCompanys;

    }
    public RestResponse<String> importExcel(MultipartFile multipartfile, HttpServletRequest request){
        try {
            List<String[]> readExcel = ExcelImport.getExcelData(multipartfile);
            if(readExcel.size()<1){
                return RestResponse.codeAndMessage(1001,"导入不能没数据！！！");
            }
            List<BranchCompanyStockInfo> list=new ArrayList<>();
            for (int i = 1; i < readExcel.size(); i++) {
                String[] str = readExcel.get(i);
                BranchCompany branchCompany=new BranchCompany();
                branchCompany.setName(str[0]);
                branchCompany=mapper.selectOne(branchCompany);
                if(branchCompany!=null){
                    BranchCompany branchCompany1=new BranchCompany();
                    if(str.length>1){
                        branchCompany1.setLeader(str[1]);
                    }
                    if(str.length>2){
                        branchCompany1.setLeaderContactInfo(str[2]);
                    }
                   if(branchCompany1!=null&&(StringUtils.isNotBlank(branchCompany1.getLeader())||StringUtils.isNotBlank(branchCompany1.getLeaderContactInfo()))){
                       branchCompany1.setId(branchCompany.getId());
                       mapper.updateByPrimaryKeySelective(branchCompany1);
                   }

                }
            }
        }catch (Exception e){
            e.printStackTrace();
            return RestResponse.codeAndMessage(10001,"网络异常！");
        }
        return RestResponse.suc();
    }

    public List<BranchCompany> getListByUser(UserDTO userDTO){
        if(userDTO != null && DATA_ALL_FALSE.equals(userDTO.getDataAll())) {
            Example example = new Example(BranchCompany.class);
            Example.Criteria criteria = example.createCriteria();
            if (StringUtils.isNotBlank(userDTO.getDataZone())) {
                criteria.andIn(COMPANY_ZONE_ID, userDTO.dataZone2List());
            }
            if (StringUtils.isNotBlank(userDTO.getDataCompany())) {
                criteria.andIn(ID, userDTO.dataCompany2List());
            }
            example.setOrderByClause("`id` asc");
            return mapper.selectByExample(example);
        }
        return  getAll();

    }

}
