package com.xxfc.platform.vehicle.biz;

import com.github.wxiaoqi.security.admin.entity.BaseUserMember;
import com.github.wxiaoqi.security.admin.entity.BaseUserMemberLevel;
import com.github.wxiaoqi.security.admin.feign.UserFeign;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.xxfc.platform.vehicle.entity.VehicleModelCalendarPrice;
import com.xxfc.platform.vehicle.mapper.VehicleModelCalendarPriceMapper;
import com.xxfc.platform.vehicle.pojo.dto.VehicleModelCalendarPriceDTO;
import com.xxfc.platform.vehicle.pojo.dto.VehicleModelCalendarPriceSaveDTO;
import com.xxfc.platform.vehicle.pojo.dto.VehicleModelDTO;
import com.xxfc.platform.vehicle.pojo.dto.VehicleModelHolidayPriceDTO;
import com.xxfc.platform.vehicle.pojo.vo.VehicleModelDayPriceVo;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import tk.mybatis.mapper.entity.Example;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.time.ZoneId;
import java.time.temporal.TemporalAdjusters;
import java.util.*;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * @author libin
 * @version 1.0
 * @description
 * @data 2019/10/14 17:32
 */
@Transactional(rollbackFor = Exception.class)
@Service
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
@Slf4j
public class VehicleModelCalendarPriceBiz extends BaseBiz<VehicleModelCalendarPriceMapper, VehicleModelCalendarPrice> {
    private static final int START_OF_WEEK = 1;
    private static final int END_OF_WEEK = 7;
    private static final Integer DEFAULT_DISCOUNT = 100;
    private static final Integer DEFAULT_FREE_DAYS = 1;
    private static final Integer DEFAULT_MEMBER_LEVEL = 0;
    private static final String PRICE_VAL = "price";
    private static final String BASE_PRICE_VAL="basePrice";
    private static final String DAYS_VAL = "freeDays";

    private final VehicleModelHolidayPriceBiz vehicleModelHolidayPriceBiz;
    private final VehicleModelBiz vehicleModelBiz;
    private final UserFeign userFeign;


    /**
     * 保存
     *
     * @param vehicleModelCalendarPrices
     * @param userId
     */
    public void addVehicleModelCalendarPrice(List<VehicleModelCalendarPriceSaveDTO> vehicleModelCalendarPrices, Integer userId) {
        if (CollectionUtils.isNotEmpty(vehicleModelCalendarPrices)) {
            List<VehicleModelCalendarPrice> vehicleModelCalendarPriceList = new ArrayList<>();
            List<Date> dateList = vehicleModelCalendarPrices.stream().peek(x -> {
                VehicleModelCalendarPrice calendarPrice = new VehicleModelCalendarPrice();
                BeanUtils.copyProperties(x, calendarPrice);
                Date date  = localDateToDate(LocalDate.parse(x.getDate()));
                calendarPrice.setVehicleModelDay(date);
                calendarPrice.setCrtTime(new Date());
                calendarPrice.setCrtUserId(userId);
                calendarPrice.setIsDel(0);
                x.setVehicleModelDay(date);
                vehicleModelCalendarPriceList.add(calendarPrice);

            }).map(VehicleModelCalendarPriceSaveDTO::getVehicleModelDay).distinct().collect(Collectors.toList());

            //1.删除当月做更改了的日期的数据
            VehicleModelCalendarPrice vehicleModelCalendarPrice = new VehicleModelCalendarPrice();
            vehicleModelCalendarPrice.setUpdUserId(userId);
            vehicleModelCalendarPrice.setUpdTime(new Date());
            vehicleModelCalendarPrice.setIsDel(1);
            Example example = new Example(VehicleModelCalendarPrice.class);
            Example.Criteria criteria = example.createCriteria();
            criteria.andIn("vehicleModelDay", dateList);
            mapper.updateByExampleSelective(vehicleModelCalendarPrice, example);
            //2.插入新的数据
            mapper.insertList(vehicleModelCalendarPriceList);
        }

    }

    /**
     * 查询全部车型
     *
     * @return
     */
    public List<VehicleModelDayPriceVo> listVehicleModelPrice() {
        List<VehicleModelDayPriceVo> vehicleModelDayPriceVos = new ArrayList<>();
        List<VehicleModelDTO> vehicleModelDTOS = vehicleModelBiz.findAllVehicleModel();
        VehicleModelDayPriceVo vehicleModelDayPriceVo;
        for (VehicleModelDTO vehicleModelDTO : vehicleModelDTOS) {
            vehicleModelDayPriceVo = new VehicleModelDayPriceVo();
            BeanUtils.copyProperties(vehicleModelDTO, vehicleModelDayPriceVo);
            vehicleModelDayPriceVos.add(vehicleModelDayPriceVo);
        }
        return vehicleModelDayPriceVos;
    }

    /**
     * 根据俱体日期查询
     *
     * @param currentDate
     * @return
     */
    public List<VehicleModelDayPriceVo> findVehicleModelcalendarPriceByDateWithDay(Date currentDate) {
        List<VehicleModelDayPriceVo> vehicleModelDayPriceVos = new ArrayList<>();
        Map<Integer, VehicleModelCalendarPrice> vehicleModelCalendarPriceMap = null;
        Example example = new Example(VehicleModelCalendarPrice.class);
        Example.Criteria criteria = example.createCriteria();
        criteria.andEqualTo("vehicleModelDay", currentDate);
        criteria.andEqualTo("isDel", 0);
        List<VehicleModelCalendarPrice> vehicleModelCalendarPriceList = mapper.selectByExample(example);
        if (CollectionUtils.isNotEmpty(vehicleModelCalendarPriceList)) {
            vehicleModelCalendarPriceMap = vehicleModelCalendarPriceList.stream().collect(Collectors.toMap(VehicleModelCalendarPrice::getVehicleModelId, Function.identity()));
        }
        List<VehicleModelDTO> vehicleModelDTOS = vehicleModelBiz.findAllVehicleModel();
        VehicleModelDayPriceVo vehicleModelDayPriceVo;
        for (VehicleModelDTO vehicleModelDTO : vehicleModelDTOS) {
            vehicleModelDayPriceVo = new VehicleModelDayPriceVo();
            BeanUtils.copyProperties(vehicleModelDTO, vehicleModelDayPriceVo);
            boolean isNullOfVehicleModelPrice = vehicleModelCalendarPriceMap == null ? true : vehicleModelCalendarPriceMap.get(vehicleModelDTO.getVehicleModelId()) == null;
            Double multiple = isNullOfVehicleModelPrice ? null : vehicleModelCalendarPriceMap.get(vehicleModelDTO.getVehicleModelId()).getMultiple();
            vehicleModelDayPriceVo.setMultiple(multiple);
            BigDecimal aPrice = isNullOfVehicleModelPrice ? null : vehicleModelCalendarPriceMap.get(vehicleModelDTO.getVehicleModelId()).getPrice();
            vehicleModelDayPriceVo.setPrice(aPrice);
            Integer level =  isNullOfVehicleModelPrice?null:vehicleModelCalendarPriceMap.get(vehicleModelDTO.getVehicleModelId()).getLevel();
            vehicleModelDayPriceVo.setLevel(level);
            vehicleModelDayPriceVos.add(vehicleModelDayPriceVo);
        }
        return vehicleModelDayPriceVos;
    }

    /**
     * 根据月份查询
     *
     * @param currentDate
     * @return
     */
    public List<VehicleModelCalendarPriceSaveDTO> findVehicleModelCalendarPricesByDateWithMonth(Date currentDate) {
        List<VehicleModelCalendarPriceSaveDTO> vehicleModelCalendarPriceSaveDTOS = new ArrayList<>();
        Example example = new Example(VehicleModelCalendarPrice.class);
        Example.Criteria criteria = example.createCriteria();
        LocalDate startLocalDate = dateToLocalDate(currentDate).withDayOfMonth(1);
        Date startDate = localDateToDate(startLocalDate);
        LocalDate endLocalDate = startLocalDate.with(TemporalAdjusters.lastDayOfMonth());
        Date endDate = localDateToDate(endLocalDate);
        criteria.andBetween("vehicleModelDay", startDate, endDate);
        criteria.andEqualTo("isDel", 0);
        List<VehicleModelCalendarPrice> vehicleModelCalendarPriceList = mapper.selectByExample(example);
        if (CollectionUtils.isEmpty(vehicleModelCalendarPriceList)) {
            return vehicleModelCalendarPriceSaveDTOS;
        }

        VehicleModelCalendarPriceSaveDTO vehicleModelCalendarPrice;
        for (VehicleModelCalendarPrice calendarPrice : vehicleModelCalendarPriceList) {
            vehicleModelCalendarPrice = new VehicleModelCalendarPriceSaveDTO();
            BeanUtils.copyProperties(calendarPrice, vehicleModelCalendarPrice);
            vehicleModelCalendarPrice.setVehicleModelId(null);
            vehicleModelCalendarPriceSaveDTOS.add(vehicleModelCalendarPrice);
        }

        Map<Date, Optional<VehicleModelCalendarPriceSaveDTO>> result = vehicleModelCalendarPriceSaveDTOS.stream()
                .collect(Collectors.groupingBy(VehicleModelCalendarPriceSaveDTO::getVehicleModelDay, Collectors.maxBy(Comparator.comparing(VehicleModelCalendarPriceSaveDTO::getId))));
        vehicleModelCalendarPriceSaveDTOS =  result.values().stream().map(x->x.orElseGet(null)).filter(Objects::nonNull).collect(Collectors.toList());
        return vehicleModelCalendarPriceSaveDTOS;
    }

    /**
     * 根据下单时间查询
     *
     * @param startDate
     * @param endDate
     * @param vehicleModelId
     * @param userId
     * @return
     */
    public List<VehicleModelCalendarPriceDTO> findVehicleModelCalendarPriceByDateAndVehilceIdAndUserId(Date startDate, Date endDate, Integer vehicleModelId, Integer userId) {
        return findVehicleModelCalendarPrice(startDate, endDate, vehicleModelId, userId);
    }

    /**
     * 日历展示车型价格
     *
     * @param startDate
     * @param endDate
     * @param vehicleModelId
     * @param userId
     * @return
     */
    public List<VehicleModelCalendarPriceDTO> listVehicleModelCalendarPriceByDateAndVehicleModelIdAndUserId(Date startDate, Date endDate, Integer vehicleModelId, Integer userId) {
        LocalDate startLocalDate = startDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        LocalDate endLocalDate = endDate.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();

        //处理后延伸的开始时间
        AtomicReference<Date> startReference = new AtomicReference<>(startDate);
        //处理后延伸的结束时间
        AtomicReference<Date> endReference = new AtomicReference<>(endDate);
        transformStartDateAndEndDate(startReference, endReference);
        List<VehicleModelCalendarPriceDTO> vehicleModelCalendarPrice = findVehicleModelCalendarPrice(startReference.get(), endReference.get(), vehicleModelId, userId);
        for (VehicleModelCalendarPriceDTO vehicleModelCalendarPriceDTO : vehicleModelCalendarPrice) {
            LocalDate current_date = dateToLocalDate(vehicleModelCalendarPriceDTO.getDate());
            boolean isSelect = (current_date.isAfter(startLocalDate) && current_date.isBefore(endLocalDate)) || current_date.isEqual(startLocalDate) || current_date.isEqual(endLocalDate);
            vehicleModelCalendarPriceDTO.setIsSelect(isSelect ? true : false);
        }
        return vehicleModelCalendarPrice;
    }

    /**
     * 日历价格查询
     *
     * @param startDate
     * @param endDate
     * @param vehicleModelId
     * @return
     */
    private List<VehicleModelCalendarPriceDTO> findVehicleModelCalendarPrice(Date startDate, Date endDate, Integer vehicleModelId, Integer userId) {
        List<VehicleModelCalendarPriceDTO> vehicleModelCalendarPriceVos = new ArrayList<>();
        //默认折扣
        Integer discount = DEFAULT_DISCOUNT;
        //会员默认等级
        Integer memberLevel = DEFAULT_MEMBER_LEVEL;
        //日历价格转map
        List<VehicleModelCalendarPrice> vehicleModelCalendarPrices = getVehicleModelCalendarPricesByVehicleModelIdAndDate(vehicleModelId, startDate, endDate);
        Map<Date, VehicleModelCalendarPrice> calendarPriceMap = vehicleModelCalendarPrices.stream().collect(Collectors.toMap(VehicleModelCalendarPrice::getVehicleModelDay, Function.identity()));
        //节假日转map
        List<VehicleModelHolidayPriceDTO> vehicleModelHolidayPrices = vehicleModelHolidayPriceBiz.findVehicleModelHolidayPriceByMonth(startDate, endDate);
        Map<Date, VehicleModelHolidayPriceDTO> festivalDayMap = vehicleModelHolidayPrices.stream().collect(Collectors.toMap(VehicleModelHolidayPriceDTO::getFestivalDay, Function.identity()));
        Map<Integer, Integer> levelAndDiscountMap = userFeign.levels().stream().collect(Collectors.toMap(BaseUserMemberLevel::getLevel, BaseUserMemberLevel::getDiscount));

        //车型的原价
        BigDecimal vehicle_base_price = new BigDecimal(0);
        BigDecimal vehicle_price = vehicleModelBiz.findVehicleModelPriceByVehicleModelId(vehicleModelId);
        vehicle_base_price = vehicle_base_price.add(vehicle_price);
        if (Objects.nonNull(userId)) {
            BaseUserMember baseUserMember = userFeign.findBaseUserMemberByUserId(userId);
            discount = baseUserMember == null ? discount : baseUserMember.getDiscount();
            memberLevel = baseUserMember == null ? memberLevel : baseUserMember.getMemberLevel();
        }
        VehicleModelCalendarPriceDTO vehicleModelCalendarPriceDTO;
        LocalDate final_startLocalDate = dateToLocalDate(startDate);
        LocalDate final_endLocalDate = dateToLocalDate(endDate);
        while (final_startLocalDate.isBefore(final_endLocalDate) || final_startLocalDate.isEqual(final_endLocalDate)) {
            vehicleModelCalendarPriceDTO = new VehicleModelCalendarPriceDTO();
            //价格重置
            vehicle_price = vehicle_base_price;
            //未乘以会员折扣的价格
            BigDecimal no_discount_price = new BigDecimal(vehicle_price.doubleValue());
            //免费天数重置
            Integer free_days = DEFAULT_FREE_DAYS;
            //节假日对应的价格和免费天数
            Map<String, Object> price_freeDays_map = null;
            Date current_date = localDateToDate(final_startLocalDate);
            vehicleModelCalendarPriceDTO.setDate(current_date);
            if (calendarPriceMap != null && !calendarPriceMap.isEmpty()) {
                VehicleModelCalendarPrice vehicleModelCalendarPrice = calendarPriceMap.get(current_date);
                if (Objects.isNull(vehicleModelCalendarPrice)) {
                    price_freeDays_map = transfromPriceAndFreeDaysByDate(festivalDayMap, current_date, vehicle_base_price, discount);
                } else {
                    switch (vehicleModelCalendarPrice.getType()) {
                        case VehicleModelPriceType.MULTIPLE:
                            no_discount_price = vehicle_price.multiply(new BigDecimal(vehicleModelCalendarPrice.getMultiple().doubleValue()));
                            vehicle_price = no_discount_price.multiply(new BigDecimal(discount / 100.00));
                            break;
                        case VehicleModelPriceType.ABS:
                            no_discount_price = vehicleModelCalendarPrice.getPrice();
                            vehicle_price = no_discount_price.multiply(new BigDecimal(discount / 100.00));
                            break;
                        case VehicleModelPriceType.MEMBER:
                            memberLevel = vehicleModelCalendarPrice.getLevel() > memberLevel ? vehicleModelCalendarPrice.getLevel() : memberLevel;
                            Integer level_discount = levelAndDiscountMap.get(memberLevel);
                            vehicle_price = level_discount == null ? vehicle_price : vehicle_price.multiply(new BigDecimal(level_discount / 100.00));
                            break;
                        default:
                            break;
                    }
                    free_days = vehicleModelCalendarPrice.getFreeDays() == null ? free_days : vehicleModelCalendarPrice.getFreeDays();
                }
            } else {
                price_freeDays_map = transfromPriceAndFreeDaysByDate(festivalDayMap, current_date, vehicle_base_price, discount);
            }
            if (price_freeDays_map != null && !price_freeDays_map.isEmpty()) {
                vehicle_price = (BigDecimal) price_freeDays_map.get(PRICE_VAL);
                free_days = (Integer) price_freeDays_map.get(DAYS_VAL);
                no_discount_price = (BigDecimal) price_freeDays_map.get(BASE_PRICE_VAL);
            }
            vehicleModelCalendarPriceDTO.setNo_discount_price(no_discount_price);
            vehicleModelCalendarPriceDTO.setPrice(vehicle_price.setScale(2, RoundingMode.HALF_UP));
            vehicleModelCalendarPriceDTO.setFreeDays(free_days);
            final_startLocalDate = final_startLocalDate.plusDays(1);
            vehicleModelCalendarPriceVos.add(vehicleModelCalendarPriceDTO);
        }
        return vehicleModelCalendarPriceVos;
    }

    /**
     * (节假日|非节假日未设置的)价格和免费天数 处理
     *
     * @param festivalDayMap
     * @param current_date
     * @param vehicle_price
     * @param discount
     * @return
     */
    public Map<String, Object> transfromPriceAndFreeDaysByDate(Map<Date, VehicleModelHolidayPriceDTO> festivalDayMap, Date current_date, BigDecimal vehicle_price, Integer discount) {
        log.info("参数：【{}=={}=={}=={}】",festivalDayMap,current_date,vehicle_price,discount);
        Map<String, Object> vehicle_price_days_map = new HashMap<>(3);
        Integer free_days = DEFAULT_FREE_DAYS;
        vehicle_price_days_map.put(BASE_PRICE_VAL,vehicle_price);
        if (festivalDayMap != null && !festivalDayMap.isEmpty()) {
            VehicleModelHolidayPriceDTO vehicleModelHolidayPriceDTO = festivalDayMap.get(current_date);
            if (Objects.nonNull(vehicleModelHolidayPriceDTO)) {
                vehicle_price = vehicle_price.multiply(new BigDecimal(vehicleModelHolidayPriceDTO.getMultiple().doubleValue()));
                free_days = vehicleModelHolidayPriceDTO.getFreeDays() == null ? free_days : vehicleModelHolidayPriceDTO.getFreeDays();
            }
        }
        vehicle_price = vehicle_price.multiply(new BigDecimal(discount / 100.00));
        vehicle_price_days_map.put(PRICE_VAL, vehicle_price);
        vehicle_price_days_map.put(DAYS_VAL, free_days);
        log.info("(节假日|非节假日未设置的)价格和免费天数 处理【{}】",vehicle_price_days_map);
        return vehicle_price_days_map;
    }

    /**
     * 根据时间范围查询
     *
     * @param vehicleModelId
     * @param final_startDate
     * @param final_endDate
     * @return
     */
    private List<VehicleModelCalendarPrice> getVehicleModelCalendarPricesByVehicleModelIdAndDate(Integer vehicleModelId, Date final_startDate, Date final_endDate) {
        Example example = new Example(VehicleModelCalendarPrice.class);

        Example.Criteria base_criteria = example.createCriteria();
        base_criteria.andEqualTo("isDel", 0);
        base_criteria.andBetween("vehicleModelDay", final_startDate, final_endDate);

        Example.Criteria vehicle_criteria = example.createCriteria();
        vehicle_criteria.andEqualTo("vehicleModelId", vehicleModelId);
        vehicle_criteria.orIsNull("vehicleModelId");
        example.and(vehicle_criteria);

        List<VehicleModelCalendarPrice> vehicleModelCalendarPrices = mapper.selectByExample(example);
        return CollectionUtils.isEmpty(vehicleModelCalendarPrices) ? Collections.EMPTY_LIST : vehicleModelCalendarPrices;
    }

    /**
     * 时间处理
     *
     * @param startDatee
     * @param endDatee
     */
    public static void transformStartDateAndEndDate(AtomicReference<Date> startDatee, AtomicReference<Date> endDatee) {
        Date startDate = startDatee.get();
        Date endDate = endDatee.get();
        if (startDate == null && endDate == null) {
            LocalDate now = LocalDate.now();
            LocalDate endLocalDate = now.with(TemporalAdjusters.lastDayOfMonth());
            LocalDate startLocalDate = now.withDayOfMonth(1);
            startDate = localDateToDate(startLocalDate);
            endDate =  localDateToDate(endLocalDate);

        } else {
            int days = 0;
            /****************************************开始时间处理******************************************************/
            LocalDate startLocalDate = dateToLocalDate(startDate);
            int start_week = startLocalDate.getDayOfWeek().getValue();
            if (START_OF_WEEK < start_week && start_week < END_OF_WEEK) {
                days =start_week - START_OF_WEEK;
            } else {
                if (END_OF_WEEK == start_week) {
                    days = END_OF_WEEK-1;
                }
            }
            LocalDate start_startLocalDate = startLocalDate.minusDays(days);

            /****************************************结束时间处理******************************************************/
            days = 0;
            LocalDate endLocalDate = dateToLocalDate(endDate);
            int end_week = endLocalDate.getDayOfWeek().getValue();

            if (START_OF_WEEK < end_week && end_week < END_OF_WEEK) {
                days = END_OF_WEEK - end_week;
            } else {
                if (START_OF_WEEK == end_week) {
                    days =END_OF_WEEK - 1;
                }
            }
            LocalDate end_endLocalDate = endLocalDate.plusDays(days);
            /****************************************时间转换******************************************************/
            startDate = localDateToDate(start_startLocalDate);
            endDate = localDateToDate(end_endLocalDate);
        }
        startDatee.set(startDate);
        endDatee.set(endDate);
    }

    private static LocalDate dateToLocalDate(Date date){
      return LocalDate.from(date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate());
    }

    private static Date  localDateToDate(LocalDate localDate){
      return Date.from(localDate.atStartOfDay(ZoneId.systemDefault()).toInstant());
    }

    public static void main(String[] args) {
        AtomicReference<Date> start = new AtomicReference<>(Date.from(LocalDate.parse("2019-10-03").atStartOfDay(ZoneId.systemDefault()).toInstant()));
        AtomicReference<Date> end = new AtomicReference<>(Date.from(LocalDate.parse("2019-10-29").atStartOfDay(ZoneId.systemDefault()).toInstant()));

        transformStartDateAndEndDate(start,end);
        System.out.println(start.get());
        System.out.println(end.get());
    }

    /**
     * 车型日历价格设置类型
     */
    private class VehicleModelPriceType {
        /**
         * 倍数
         */
        private static final int MULTIPLE = 1;
        /**
         * 绝对值
         */
        private static final int ABS = 2;
        /**
         * 会员价
         */
        private static final int MEMBER = 3;
    }
}
