package com.github.wxiaoqi.security.admin.biz;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.map.MapUtil;
import cn.hutool.poi.excel.ExcelUtil;
import cn.hutool.poi.excel.ExcelWriter;
import com.github.wxiaoqi.security.admin.dto.MemberShareRecordDto;
import com.github.wxiaoqi.security.admin.entity.BaseUserMember;
import com.github.wxiaoqi.security.admin.entity.BaseUserMemberLevel;
import com.github.wxiaoqi.security.admin.entity.MemberShareCountRecord;
import com.github.wxiaoqi.security.admin.entity.MemberShareRecord;
import com.github.wxiaoqi.security.admin.mapper.BaseUserMemberLevelMapper;
import com.github.wxiaoqi.security.admin.mapper.MemberShareRecordMapper;
import com.github.wxiaoqi.security.admin.util.DateUtils;
import com.github.wxiaoqi.security.admin.vo.AppUserVo;
import com.github.wxiaoqi.security.admin.vo.MemberShareRecordVo;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.github.wxiaoqi.security.common.msg.ObjectRestResponse;
import com.github.wxiaoqi.security.common.util.Query;
import com.github.wxiaoqi.security.common.vo.PageDataVO;
import com.google.common.base.Predicates;
import com.google.common.collect.Iterables;
import com.xxfc.platform.order.entity.OrderMemberDetail;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import tk.mybatis.mapper.entity.Example;

import javax.servlet.http.HttpServletResponse;
import java.math.BigDecimal;
import java.net.URLEncoder;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 会员分成记录
 *
 * @author jiaoruizhen
 * @email jiaoruizhen@126.com
 * @date 2020-11-03 11:12:15
 */
@Service
@Slf4j
public class MemberShareRecordBiz extends BaseBiz<MemberShareRecordMapper, MemberShareRecord> {

	@Autowired
	MemberShareCountRecordBiz memberShareCountRecordBiz;
	@Autowired
	BaseUserMemberBiz baseUserMemberBiz;
	@Autowired
	BaseUserMemberLevelMapper baseUserMemberLevelMapper;
	@Autowired
	AppUserDetailBiz appUserDetailBiz;

	/**
	 * 根据订单号查询分成记录
	 * 每个订单有且只有一条记录
	 *
	 * @param no
	 * @return
	 */
	public MemberShareRecord getOneByOrderNo(String no) {
		Example example = new Example(MemberShareRecord.class);
		example.createCriteria().andEqualTo("orderNo", no);
		return mapper.selectOneByExample(example);
	}

	/**
	 * 新增或者编辑会员分成记录
	 *
	 * @param memberShareRecord
	 */
	public void addUpdate(MemberShareRecord memberShareRecord) {
		if (memberShareRecord != null) {
			MemberShareRecord old = getOneByOrderNo(memberShareRecord.getOrderNo());
			if (old == null) {
				insertSelectiveRe(memberShareRecord);
				updateAllLevelInfo(memberShareRecord);
			} else {
				if (StringUtils.isNotBlank(memberShareRecord.getMemberIds())) {
					old.setMemberIds(memberShareRecord.getMemberIds());
				}
				if (memberShareRecord.getMemberNum() != null) {
					old.setMemberNum(memberShareRecord.getMemberNum());
				}
				if (memberShareRecord.getAmount() != null) {
					old.setAmount(memberShareRecord.getAmount());
				}
				updateSelectiveByIdRe(old);
				updateAllLevelInfo(old);
			}
		}
		generateData();
	}

	/**
	 * 查看某一用户分成记录明细
	 *
	 * @return
	 */
	public ObjectRestResponse<List<MemberShareRecord>> getAllRecord(MemberShareRecordDto memberShareRecordDto) {
		Query query = new Query(memberShareRecordDto);
		PageDataVO<MemberShareRecord> pageDataVO = PageDataVO.pageInfo(query, () -> mapper.selectList(query.getSuper()));
		return ObjectRestResponse.succ(pageDataVO);
	}

	/**
	 * 更新相同会员等级的分成信息
	 *
	 * @param memberShareRecord
	 */
	public void updateAllLevelInfo(MemberShareRecord memberShareRecord) {
		if (memberShareRecord != null && memberShareRecord.getMemberLevel() != null) {
			List<MemberShareRecord> list = getAllByLevel(memberShareRecord.getMemberLevel());
			if (list != null && list.size() > 0) {
				list.parallelStream().forEach(record -> {
					if (StringUtils.isNotBlank(memberShareRecord.getMemberIds())) {
						record.setMemberIds(memberShareRecord.getMemberIds());
					}
					if (memberShareRecord.getMemberNum() != null) {
						record.setMemberNum(memberShareRecord.getMemberNum());
					}
					if (memberShareRecord.getAmount() != null) {
						record.setAmount(memberShareRecord.getAmount());
					}
					updateSelectiveByIdRe(record);
				});
			}
		}
	}

	/**
	 * 获取同级别的所有分成记录
	 *
	 * @param level
	 * @return
	 */
	private List<MemberShareRecord> getAllByLevel(Integer level) {
		Example example = new Example(MemberShareRecord.class);
		example.createCriteria().andEqualTo("memberLevel", level);
		return mapper.selectByExample(example);
	}

	/**
	 * 根据日期查询所有分成记录
	 *
	 * @param startDate
	 * @param endDate
	 * @return
	 */
	private List<MemberShareRecord> getAllByDate(Long startDate, Long endDate) {
		Example example = new Example(MemberShareRecord.class);
		example.createCriteria().andBetween("payTime", startDate, endDate);
		return mapper.selectByExample(example);
	}

	/**
	 * 统计分成信息
	 */
	public void generateData() {
		DateTime dateTime = DateTime.now().minusMonths(1);
		Long startDate = DateUtils.getStartOfDay(dateTime.toDate());
		Long endDate = DateUtils.getEndOfDay(dateTime.toDate());
		List<MemberShareRecord> list = getAllByDate(startDate, endDate);
		if (list != null && list.size() > 0) {
			Map<Integer, List<MemberShareRecord>> groupedMap =
					list.stream().collect(Collectors.groupingBy(MemberShareRecord::getMemberLevel));
			if (MapUtil.isNotEmpty(groupedMap)) {
				for (Map.Entry<Integer, List<MemberShareRecord>> entry : groupedMap.entrySet()) {
					Integer level = entry.getKey();
					List<MemberShareRecord> memberShareRecordList = entry.getValue();
					generateDataByUser(memberShareRecordList, level, dateTime);
				}
			}
		}
	}


	private void generateDataByUser(List<MemberShareRecord> memberShareRecordList, Integer level, DateTime dateTime) {
		if (memberShareRecordList != null) {
			Map<Long, BigDecimal> amountMap = new HashMap<>();
			memberShareRecordList.parallelStream().forEach(memberShareRecord -> {
				if (StringUtils.isNotBlank(memberShareRecord.getMemberIds())) {
					String[] arr = memberShareRecord.getMemberIds().split(",");
					if (arr != null) {
						for (String id : arr) {
							if (StringUtils.isNotBlank(id)) {
								Long userId = Long.parseLong(id);
								if (amountMap.get(userId) != null) {
									amountMap.put(userId, amountMap.get(userId).add(memberShareRecord.getAmount()));
								} else {
									amountMap.put(userId, memberShareRecord.getAmount());
								}
							}
						}
					}
				}
			});

			if (MapUtil.isNotEmpty(amountMap)) {
				List<MemberShareCountRecord> memberShareCountRecordList = new ArrayList<>();
				for (Map.Entry<Long, BigDecimal> entry : amountMap.entrySet()) {
					MemberShareCountRecord memberShareCountRecord = new MemberShareCountRecord();
					memberShareCountRecord.setUserId(entry.getKey());
					memberShareCountRecord.setAmount(entry.getValue());
					memberShareCountRecord.setMemberLevel(level);
					memberShareCountRecord.setCountMonth(dateTime.toString("yyyy-MM"));
					memberShareCountRecord.setCountDay(dateTime.toString("dd"));
					memberShareCountRecord.setIssue(0);
					memberShareCountRecord.setCrtTime(System.currentTimeMillis());
					memberShareCountRecord.setUpdTime(System.currentTimeMillis());
					memberShareCountRecordList.add(memberShareCountRecord);
				}

				if (memberShareCountRecordList.size() > 0) {
					memberShareCountRecordBiz.batchSave(memberShareCountRecordList);
				}

			}

		}
	}

	/**
	 * 接收会员订单支付成功后处理
	 *
	 * @param orderMemberDetail
	 */
	public void generateMemberShareQueue(OrderMemberDetail orderMemberDetail) {
		if (orderMemberDetail != null) {
			log.info("会员订单信息: {}", orderMemberDetail);
			Example example = new Example(BaseUserMemberLevel.class);
			example.createCriteria().andEqualTo("isdel", 0).andEqualTo("isShare", 1);
			example.orderBy("level");
			List<BaseUserMemberLevel> baseUserMemberLevelList = baseUserMemberLevelMapper.selectByExample(example);
			if (baseUserMemberLevelList != null && baseUserMemberLevelList.size() > 0) {
				//如果购买会员是分成会员低一级会员以及分成会员，才参与分成
				if (orderMemberDetail.getMemberLevel().compareTo(baseUserMemberLevelList.get(0).getLevel() - 1) >= 0) {
					//所有分成会员都需要新增一条记录
					for (BaseUserMemberLevel baseUserMemberLevel : baseUserMemberLevelList) {
						List<BaseUserMember> baseUserMemberList = baseUserMemberBiz.selectList(new BaseUserMember() {{
							setIsDel(0);
							setMemberLevel(baseUserMemberLevel.getLevel());
						}});
						if (baseUserMemberList != null && baseUserMemberList.size() > 0) {
							List<Integer> userIds = baseUserMemberList.stream().map(BaseUserMember::getUserId).collect(Collectors.toList());
							if (userIds != null && userIds.size() > 0) {
								MemberShareRecord memberShareRecord = new MemberShareRecord();
								memberShareRecord.setOrderNo(orderMemberDetail.getBaseOrder().getNo());
								memberShareRecord.setPayTime(orderMemberDetail.getBaseOrder().getPayTime());
								memberShareRecord.setOrderAmount(orderMemberDetail.getBaseOrder().getOrderAmount());
								memberShareRecord.setMemberLevel(baseUserMemberLevel.getLevel());
								memberShareRecord.setMemberLevelId(baseUserMemberLevel.getId());
								memberShareRecord.setUserId(orderMemberDetail.getBaseOrder().getUserId().longValue());
								memberShareRecord.setOrderTime(orderMemberDetail.getCrtTime());
								memberShareRecord.setTotalAmount(orderMemberDetail.getBaseOrder().getOrderAmount().multiply(baseUserMemberLevel.getShareAmount()).divide(new BigDecimal(100)).setScale(2, BigDecimal.ROUND_HALF_UP));
								memberShareRecord.setMemberNum(userIds.size());
								memberShareRecord.setAmount(baseUserMemberLevel.getShareAmount().divide(new BigDecimal(userIds.size()),2, BigDecimal.ROUND_HALF_UP));
								StringBuilder stringBuilder = new StringBuilder();
								userIds.parallelStream().forEach(id -> {
									stringBuilder.append(",");
									stringBuilder.append(id);
									stringBuilder.append(",");
								});
								memberShareRecord.setMemberIds(stringBuilder.toString());
								addUpdate(memberShareRecord);
							}
						}
					}
				}
			}
		}
	}

	/**
	 * 会员分成导出信息
	 *
	 * @param httpServletResponse
	 * @param memberShareRecordDto
	 * @throws Exception
	 */
	public void exportRecordList(HttpServletResponse httpServletResponse, MemberShareRecordDto memberShareRecordDto) throws Exception {

		AppUserVo appUserDetail = appUserDetailBiz.getUserInfoById(memberShareRecordDto.getUserId());
		if (appUserDetail != null) {
			ExcelWriter writer = null;
			try {
				writer = ExcelUtil.getWriter(true);
				//封装表头
				writer.addHeaderAlias("orderNo", "订单编号");
				writer.addHeaderAlias("payUser", "下单用户");
				writer.addHeaderAlias("payTime", "支付时间");
				writer.addHeaderAlias("orderAmount", "订单金额");
				writer.addHeaderAlias("totalAmount", "总分红");
				writer.addHeaderAlias("userNum", "分红人数");
				writer.addHeaderAlias("realName", "分红人");
				writer.addHeaderAlias("levelName", "等级");
				writer.addHeaderAlias("amount", "金额");
				//处理数据
				List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
				memberShareRecordDto.setLimit(Integer.MAX_VALUE);
				Query query = new Query(memberShareRecordDto);
				List<MemberShareRecord> memberShareCountRecordList = mapper.selectList(query.getSuper());
				if (memberShareCountRecordList != null && memberShareCountRecordList.size() > 0) {
					for (MemberShareRecord memberShareRecord : memberShareCountRecordList) {
						Map<String, Object> map = new HashMap();
						map.put("orderNo", memberShareRecord.getOrderNo());
						map.put("payUser", memberShareRecord.getRealName());
						map.put("payTime", DateUtil.date(memberShareRecord.getPayTime()).toString());
						map.put("orderAmount", memberShareRecord.getOrderAmount());
						map.put("totalAmount", memberShareRecord.getTotalAmount());
						map.put("userNum", memberShareRecord.getMemberNum());
						map.put("realName", appUserDetail.getRealname());
						map.put("levelName", memberShareRecord.getLevelName());
						map.put("amount", memberShareRecord.getAmount());
						list.add(map);
					}
				}
				if (list.size() == 0) {
					Map<String, Object> map = new HashMap<String, Object>();
					writer.addHeaderAlias("orderNo", "订单编号");
					writer.addHeaderAlias("payUser", "下单用户");
					writer.addHeaderAlias("payTime", "支付时间");
					writer.addHeaderAlias("orderAmount", "订单金额");
					writer.addHeaderAlias("totalAmount", "总分红");
					writer.addHeaderAlias("userNum", "分红人数");
					writer.addHeaderAlias("realName", "分红人");
					writer.addHeaderAlias("levelName", "等级");
					writer.addHeaderAlias("amount", "金额");
					map.put("orderNo", "");
					map.put("payUser", "");
					map.put("payTime", "");
					map.put("orderAmount", "");
					map.put("totalAmount", "");
					map.put("userNum", "");
					map.put("realName", "");
					map.put("levelName", "");
					map.put("amount", "");
					list.add(map);
				}
				writer.write(list);
				httpServletResponse.setContentType("application/octet-stream;charset=utf-8");
				httpServletResponse.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode("会员分成明细.xlsx", "utf-8"));
				writer.flush(httpServletResponse.getOutputStream());
			} catch (Exception ex) {
				log.error(ex.getMessage(), ex);
				throw ex;
			} finally {
				if (writer != null) {
					writer.close();
				}
			}
		}
	}

	public ObjectRestResponse selectCountList(MemberShareRecordDto memberShareRecordDto) {
		Query query = new Query(memberShareRecordDto);
		PageDataVO<MemberShareRecordVo> pageDataVO = PageDataVO.pageInfo(query, () -> mapper.selectCountList(query.getSuper()));
		if (pageDataVO != null && pageDataVO.getData() != null) {
			pageDataVO.getData().parallelStream().forEach(memberShareRecordVo -> {
				if (StringUtils.isNotBlank(memberShareRecordVo.getMemberIds())) {
					String[] arr = memberShareRecordVo.getMemberIds().split(",");
					Set<String> idList =
							new HashSet<>(Arrays.asList(arr));
					Iterables.removeIf(idList, Predicates.equalTo(""));
					memberShareRecordVo.setUserNum(idList.size());
				}
			});
		}
		return ObjectRestResponse.succ(pageDataVO);
	}

	public ObjectRestResponse<List<MemberShareRecordVo>> getAllCountRecord(MemberShareRecordDto memberShareRecordDto) {
		List<MemberShareRecordVo> memberShareRecordVoList = new ArrayList<>();
		List<MemberShareRecord> list = selectListAll();
		if (list == null || list.size() <= 0) {
			return ObjectRestResponse.succ(memberShareRecordVoList);
		}
		//按日统计
		if (memberShareRecordDto.getType() != null && memberShareRecordDto.getType() == 1) {
			generateRecordData(list, memberShareRecordVoList, 1);
			List<MemberShareRecordVo> result = memberShareRecordVoList;
			if (memberShareRecordDto.getLevel() != null) {
				result = memberShareRecordVoList.stream()
						.filter((MemberShareRecordVo s) -> s.getLevel().equals(memberShareRecordDto.getLevel()))
						.collect(Collectors.toList());
			}
			if (StringUtils.isNotBlank(memberShareRecordDto.getStartDate())) {
				result = result.stream()
						.filter((MemberShareRecordVo s) -> DateTime.parse(s.getDate(), DateTimeFormat.forPattern("yyyy-MM-dd")).compareTo(DateTime.parse(memberShareRecordDto.getStartDate(), DateTimeFormat.forPattern("yyyy-MM-dd"))) >= 0
														&& DateTime.parse(s.getDate(), DateTimeFormat.forPattern("yyyy-MM-dd")).compareTo(DateTime.parse(memberShareRecordDto.getEndDate(), DateTimeFormat.forPattern("yyyy-MM-dd"))) <= 0)
						.collect(Collectors.toList());
			}
			return ObjectRestResponse.succ(result);
		}
		//按月统计
		if (memberShareRecordDto.getType() != null && memberShareRecordDto.getType() == 2) {
			generateRecordData(list, memberShareRecordVoList, 2);
			List<MemberShareRecordVo> result = memberShareRecordVoList;
			if (memberShareRecordDto.getLevel() != null) {
				result = memberShareRecordVoList.stream()
						.filter((MemberShareRecordVo s) -> s.getLevel().equals(memberShareRecordDto.getLevel()))
						.collect(Collectors.toList());
			}
			if (StringUtils.isNotBlank(memberShareRecordDto.getStartDate())) {
				result = result.stream()
						.filter((MemberShareRecordVo s) -> DateTime.parse(s.getDate(),DateTimeFormat.forPattern("yyyy-MM")).getMillis() >= (DateTime.parse(memberShareRecordDto.getStartDate().substring(0, memberShareRecordDto.getStartDate().lastIndexOf("-")), DateTimeFormat.forPattern("yyyy-MM"))).getMillis()
								&& DateTime.parse(s.getDate(),DateTimeFormat.forPattern("yyyy-MM")).getMillis() <= (DateTime.parse(memberShareRecordDto.getEndDate().substring(0, memberShareRecordDto.getEndDate().lastIndexOf("-")), DateTimeFormat.forPattern("yyyy-MM"))).getMillis())
						.collect(Collectors.toList());
			}
			return ObjectRestResponse.succ(result);
		}
		//按年统计
		if (memberShareRecordDto.getType() != null && memberShareRecordDto.getType() == 3) {
			generateRecordData(list, memberShareRecordVoList, 3);
			List<MemberShareRecordVo> result = memberShareRecordVoList;
			if (memberShareRecordDto.getLevel() != null) {
				result = memberShareRecordVoList.stream()
						.filter((MemberShareRecordVo s) -> s.getLevel().equals(memberShareRecordDto.getType()))
						.collect(Collectors.toList());
			}
			if (StringUtils.isNotBlank(memberShareRecordDto.getStartDate())) {
				result = result.stream()
						.filter((MemberShareRecordVo s) -> DateTime.parse(s.getDate(),DateTimeFormat.forPattern("yyyy")).getMillis() >= (DateTime.parse(memberShareRecordDto.getStartDate().split("-")[0], DateTimeFormat.forPattern("yyyy"))).getMillis()
								&& DateTime.parse(s.getDate(),DateTimeFormat.forPattern("yyyy")).getMillis() <= (DateTime.parse(memberShareRecordDto.getEndDate().split("-")[0], DateTimeFormat.forPattern("yyyy"))).getMillis())
						.collect(Collectors.toList());
			}
			return ObjectRestResponse.succ(result);
		}
		return ObjectRestResponse.succ(memberShareRecordVoList);
	}


	private void generateRecordData(List<MemberShareRecord> list, List<MemberShareRecordVo> memberShareRecordVoList, Integer type) {
		List<BaseUserMemberLevel> baseUserMemberLevelList = baseUserMemberLevelMapper.select(new BaseUserMemberLevel(){{setIsdel(0);}});
		Map<Integer, List<BaseUserMemberLevel>> levelMap = baseUserMemberLevelList.stream().collect(Collectors.groupingBy(BaseUserMemberLevel::getLevel));
		Map<Integer, List<MemberShareRecord>> map = list.stream().collect(Collectors.groupingBy(MemberShareRecord::getMemberLevel));
		if (MapUtil.isNotEmpty(map)) {
			for (Map.Entry<Integer, List<MemberShareRecord>> entry : map.entrySet()) {
				MemberShareRecordVo memberShareRecordVo = new MemberShareRecordVo();
				List<BaseUserMemberLevel> baseUserMemberLevel = levelMap.get(entry.getKey());
				if (baseUserMemberLevel != null && baseUserMemberLevel.size() > 0) {
					memberShareRecordVo.setLevelName(baseUserMemberLevel.get(0).getName());
				}
				List<MemberShareRecord> memberShareRecordList = entry.getValue();
				memberShareRecordVo.setLevel(entry.getKey());
				if (memberShareRecordList != null && memberShareRecordList.size() > 0) {
					Map<String, List<MemberShareRecord>> memberMap = new HashMap<>();
					if (type == 1) {
						memberMap = memberShareRecordList.stream().collect(Collectors.groupingBy(o -> new DateTime(o.getPayTime()).toString("yyyy-MM-dd")));
					}
					if (type == 2) {
						memberMap = memberShareRecordList.stream().collect(Collectors.groupingBy(o -> new DateTime(o.getPayTime()).toString("yyyy-MM")));
					}
					if (type == 3) {
						memberMap = memberShareRecordList.stream().collect(Collectors.groupingBy(o -> new DateTime(o.getPayTime()).toString("yyyy")));
					}
					for (Map.Entry<String, List<MemberShareRecord>> entry1 : memberMap.entrySet()) {
						memberShareRecordVo.setDate(entry1.getKey());
						List<MemberShareRecord> memberShareRecords = entry1.getValue();
						if (memberShareRecords != null && memberShareRecords.size() > 0) {
							Set<String> set = new HashSet<>();
							memberShareRecords.parallelStream().forEach(memberShareRecord -> {
								if (StringUtils.isNotBlank(memberShareRecord.getMemberIds())) {
									String[] arr = memberShareRecord.getMemberIds().split(",");
									Set<String> idList =
											new HashSet<>(Arrays.asList(arr));
									Iterables.removeIf(idList, Predicates.equalTo(""));
									set.addAll(idList);
								}
							});
							BigDecimal totalAmount = memberShareRecords.stream().map(MemberShareRecord::getTotalAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
							memberShareRecordVo.setTotalAmount(totalAmount);
							memberShareRecordVo.setUserNum(set.size());
						} else {
							memberShareRecordVo.setTotalAmount(BigDecimal.ZERO);
							memberShareRecordVo.setUserNum(0);
						}
					}
				}
				memberShareRecordVoList.add(memberShareRecordVo);
			}
		}
	}

	/**
	 * 会员分成统计导出信息
	 *
	 * @param httpServletResponse
	 * @param memberShareRecordDto
	 * @throws Exception
	 */
	public void exportCountRecordList(HttpServletResponse httpServletResponse, MemberShareRecordDto memberShareRecordDto) throws Exception {
		List<MemberShareRecordVo> memberShareRecordVoList = getAllCountRecord(memberShareRecordDto).getData();
		if (memberShareRecordVoList != null) {
			ExcelWriter writer = null;
			try {
				writer = ExcelUtil.getWriter(true);
				//封装表头
				writer.addHeaderAlias("date", "统计时间");
				writer.addHeaderAlias("levelName", "会员等级");
				writer.addHeaderAlias("userNum", "会员数量");
				writer.addHeaderAlias("totalAmount", "总分红");
				//处理数据
				List<Map<String, Object>> list = new ArrayList<Map<String, Object>>();
				if (memberShareRecordVoList != null && memberShareRecordVoList.size() > 0) {
					for (MemberShareRecordVo memberShareRecord : memberShareRecordVoList) {
						Map<String, Object> map = new HashMap();
						map.put("date", memberShareRecord.getDate());
						map.put("levelName", memberShareRecord.getLevelName());
						map.put("userNum", memberShareRecord.getUserNum());
						map.put("totalAmount", memberShareRecord.getTotalAmount());
						list.add(map);
					}
				}
				if (list.size() == 0) {
					Map<String, Object> map = new HashMap<String, Object>();
					writer.addHeaderAlias("date", "统计时间");
					writer.addHeaderAlias("levelName", "会员等级");
					writer.addHeaderAlias("userNum", "会员数量");
					writer.addHeaderAlias("totalAmount", "总分红");
					map.put("date", "");
					map.put("totalAmount", "");
					map.put("userNum", "");
					map.put("levelName", "");
					list.add(map);
				}
				writer.write(list);
				httpServletResponse.setContentType("application/octet-stream;charset=utf-8");
				httpServletResponse.setHeader("Content-Disposition", "attachment;filename=" + URLEncoder.encode("会员分成明细.xlsx", "utf-8"));
				writer.flush(httpServletResponse.getOutputStream());
			} catch (Exception ex) {
				log.error(ex.getMessage(), ex);
				throw ex;
			} finally {
				if (writer != null) {
					writer.close();
				}
			}
		}
	}



	public static void main(String[] args) {
		System.out.println(DateTime.parse("2020-12-12".substring(0, "2020-11-12".lastIndexOf("-")),DateTimeFormat.forPattern("yyyy-MM")).getMillis() <= (DateTime.parse("2020-11", DateTimeFormat.forPattern("yyyy-MM"))).getMillis());
	}
}