package com.xxfc.thirdparty.service;

import com.alibaba.fastjson.JSON;
import com.xxfc.thirdparty.model.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.util.HashMap;
import java.util.Map;

/**
 * 违章查询
 */
@Service
public class TrafficViolationsService {

    @Value("${juhe.key}")
    private String KEY;

    //支持的城市
    private static final String URL_GET_CITY = "http://v.juhe.cn/wz/citys?" +
            "province={province}&dtype={dtype}&format={format}&callback={callback}&key={key}";
    //查违章
    private static final String URL_QUERY_WZ = "http://v.juhe.cn/wz/query?" +
            "dtype={dtype}&city={city}&hphm={hphm}&hpzl={hpzl}&engineno={engineno}&classno={classno}&key={key}";
    //剩余次数
    private static final String URL_BALANCE = "http://v.juhe.cn/wz/status?" +
            "dtype={dtype}&key={key}";
    //根据车牌查城市
    private static final String URL_CAR_PRE = "http://v.juhe.cn/wz/carPre?" +
            "hphm={hphm}&key={key}";

    @Autowired
    RestTemplate restTemplate;

    /**
     * 获取支持城市
     * province	string	N	默认全部，省份简写，如：ZJ、JS
     * dtype	string	N	返回数据格式：json或xml或jsonp,默认json
     * format	int	N	格式选择1或2，默认1
     * callback	String	N	返回格式选择jsonp时，必须传递
     * {"resultcode":"101","reason":"error key","result":null,"error_code":10001}
     */
    public JuheResult<Map<String, JuheTrafficViolationsProvinceInfo>> getCity(String province) {
        Map<String, String> paramsMap = new HashMap<>();
        paramsMap.put("key", KEY);
        paramsMap.put("province", province == null ? "" : province);
        paramsMap.put("dtype", "json");
        paramsMap.put("format", "");
        paramsMap.put("callback", "");
        return query(URL_GET_CITY, paramsMap);
    }

    /**
     * 查违章
     * dtype	string	否	返回数据格式：默认:json
     * city	String	是	城市代码 *
     * hphm	String	是	号牌号码 完整7位 ,需要utf8 urlencode*
     * hpzl	String	是	号牌类型，默认02:小型车，01:大型车
     * engineno	String	否	发动机号 (具体是否需要根据城市接口中的参数填写)
     * classno	String	否	车架号 (具体是否需要根据城市接口中的参数填写)
     */
    public JuheResult<JuheTrafficViolationsInfo> queryViolations(String city, String hphm, String hpzl,
                                                                 String engineno, String classno) {
        Map<String, String> paramsMap = new HashMap<>();
        paramsMap.put("key", KEY);
        paramsMap.put("dtype", "json");
        paramsMap.put("city", city);
        paramsMap.put("hphm", hphm);
        paramsMap.put("hpzl", hpzl);
        paramsMap.put("engineno", engineno);
        paramsMap.put("classno", classno);
        return query(URL_QUERY_WZ, paramsMap);
    }

    /**
     * 查询剩余次数
     */
    public JuheResult<JuheTrafficViolationsBalanceInfo> queryBalance() {
        Map<String, String> paramsMap = new HashMap<>();
        paramsMap.put("key", KEY);
        paramsMap.put("dtype", "json");
        return query(URL_BALANCE, paramsMap);
    }

    /**
     * 根据车牌前缀查城市参数
     */
    public JuheResult<JuheCarPreInfo> queryCityByHphm(String hphm) {
        Map<String, String> paramsMap = new HashMap<>();
        paramsMap.put("key", KEY);
        paramsMap.put("hphm", hphm);
        return query(URL_CAR_PRE, paramsMap);
    }

    private JuheResult query(String url, Map<String, String> paramsMap) {
        try {
            String responseStr = restTemplate.getForObject(url, String.class, paramsMap);
            return JSON.parseObject(responseStr, JuheResult.class);
        } catch (RestClientException ex) {
            JuheResult result = new JuheResult();
            result.setReason("请求错误");
            result.setResultcode("500");
            return result;
        }
    }


}
