package com.github.wxiaoqi.security.admin.biz;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.ArrayUtil;
import com.github.wxiaoqi.security.admin.bo.StaffStatisticsBo;
import com.github.wxiaoqi.security.admin.bo.StaffStatisticsTotalBo;
import com.github.wxiaoqi.security.admin.bo.UserStaffBo;
import com.github.wxiaoqi.security.admin.dto.AppUserSellingWaterDTO;
import com.github.wxiaoqi.security.admin.dto.StaffStatisticsFindDTO;
import com.github.wxiaoqi.security.admin.vo.WalletPostionVo;
import com.github.wxiaoqi.security.common.msg.ObjectRestResponse;
import com.github.wxiaoqi.security.common.vo.PageDataVO;
import com.google.common.collect.Lists;
import com.xxfc.platform.order.contant.enumerate.StatisticsStatusEnum;
import com.xxfc.platform.order.feign.OrderFeign;
import com.xxfc.platform.order.pojo.dto.OrderDTO;
import com.xxfc.platform.vehicle.feign.VehicleFeign;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;
import org.apache.poi.xssf.usermodel.XSSFCellStyle;
import org.apache.poi.xssf.usermodel.XSSFFont;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.StringUtils;

import javax.servlet.ServletOutputStream;
import java.math.BigDecimal;
import java.util.*;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author libin
 * @version 1.0
 * @description
 * @data 2019/11/26 17:01
 */
@Transactional(rollbackFor = Exception.class)
@Service
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
public class StaffStatisticsBiz {

    private static final String TOTAL_VAL = "total";
    private static final String PAGE_VAL = "page";
    private static final String TOTAL_AMOUNT_VAL = "totalAmount";
    private static final String TOTAL_NUM_VAL = "totalNum";
    private static final String TOTAL_COMMISSION_VAL = "totalCommission";
    private static final Integer NO_COMPANY_STATE = 0;
    private final AppUserDetailBiz appUserDetailBiz;
    private final AppUserSellingWaterBiz appUserSellingWaterBiz;
    private final OrderFeign orderFeign;
    private final VehicleFeign vehicleFeign;
    private final AppUserPositionBiz appUserPositionBiz;

    /**
     * 导出数据
     *
     * @param staffStatisticsFindDTO
     * @param outputStream
     */
    public void exportStaffStistics(StaffStatisticsFindDTO staffStatisticsFindDTO, ServletOutputStream outputStream) throws Exception {
        //1.查询根据员工查询
        List<UserStaffBo> userStaffBos = appUserDetailBiz.findStaffsByIdsAndCompanyAndPostion(staffStatisticsFindDTO);
        //2.计算员工的业绩
        List<StaffStatisticsBo> staffStatisticsBos = processStaffStatistics(userStaffBos, staffStatisticsFindDTO);
        //3.计算总业绩
        StaffStatisticsTotalBo staffStatisticsTotalBo = getStaffStatisticsTotalBo(userStaffBos.size(), staffStatisticsBos);

        //excel相关
        XSSFWorkbook hssfWorkbook = new XSSFWorkbook();
        XSSFSheet sheet = hssfWorkbook.createSheet("员工业绩统计报表");
        Row titleRow = sheet.createRow(0);
        CellStyle headerCellStyle = createHeaderCellStyle(hssfWorkbook);
        Cell titleCell = titleRow.createCell(0);
        titleCell.setCellValue("员工业绩统计报表");
        titleCell.setCellStyle(headerCellStyle);
        CellRangeAddress titleCellRangeAddress = new CellRangeAddress(0, 0, 0, 10);
        sheet.addMergedRegion(titleCellRangeAddress);

        Row ctpRow = sheet.createRow(1);
        Cell ctpCell = ctpRow.createCell(0);
        //
        String companyName = StringUtils.hasText(staffStatisticsFindDTO.getCompanyName()) ? staffStatisticsFindDTO.getCompanyName() : "全集团";
        String postionName = StringUtils.hasText(staffStatisticsFindDTO.getPostionName()) ? staffStatisticsFindDTO.getPostionName() : "全部员工";
        String startTimeStr = staffStatisticsFindDTO.getStartDate()==null?"":DateUtil.format(staffStatisticsFindDTO.getStartDate(), "yyyy-MM-dd HH:mm:ss");
        String endTimeStr = staffStatisticsFindDTO.getEndDate()==null?staffStatisticsFindDTO.getStartDate()==null?"":
                DateUtil.format(DateUtil.endOfDay(new Date()),"yyyy-MM-dd HH:mm:ss"):DateUtil.format(staffStatisticsFindDTO.getEndDate(), "yyyy-MM-dd HH:mm:ss");
        String ctpValue = String.format("时间:%s~%s 公司:%s 身份:%s", startTimeStr,endTimeStr,
                companyName, postionName);
        ctpCell.setCellValue(ctpValue);
        CellRangeAddress ctpCellRangeAddress = new CellRangeAddress(1, 1, 0, 10);
        sheet.addMergedRegion(ctpCellRangeAddress);

        Row totalStatisticsRow = sheet.createRow(2);
        Cell totalStatisticsRowCell = totalStatisticsRow.createCell(0);

        String totalStatisticsRowCellValue = String.format("合计 员工人数:%s 员工销售额:%s  员工App提成总额:%s", staffStatisticsTotalBo.getStaffNum(),
                staffStatisticsTotalBo.getTotalSellAmount(),
                staffStatisticsTotalBo.getTotalRoyaltyAmount());
        totalStatisticsRowCell.setCellValue(totalStatisticsRowCellValue);
        CellRangeAddress totalStatisticsRowCellRangeAddress = new CellRangeAddress(2, 2, 0, 10);
        sheet.addMergedRegion(totalStatisticsRowCellRangeAddress);

        //分割行
        CellRangeAddress lineCellRangeAddress = new CellRangeAddress(3, 3, 0, 10);
        sheet.addMergedRegion(lineCellRangeAddress);

        Row staffHeaderRow = sheet.createRow(4);
        String[] staffHeaderValue = {"序号", "员工姓名", "所属公司", "身份", "电话", "销售额(元)", "APP提成", "会员订单数/合计金额", "租车订单数/合计金额", "旅游订单数/合计金额"};
        createHeader(staffHeaderRow, 0, staffHeaderValue, headerCellStyle);
        for (int i = 0; i < staffHeaderValue.length; i++) {
            sheet.setColumnWidth(i,24*256);
        }
        //设置数据
        CellStyle generalCellStyle = createGeneralCellStyle(hssfWorkbook);
        List<String[]> staffStatisticsData = getStaffStatisticsData(staffStatisticsBos);
        createStaffStatisticsCellData(sheet, 5, generalCellStyle, staffStatisticsData);
        hssfWorkbook.write(outputStream);
        hssfWorkbook.close();
    }

    private void createStaffStatisticsCellData(Sheet sheet, int startRowIndex, CellStyle cellStyle, List<String[]> staffStatisticsData) {
        for (String[] data : staffStatisticsData) {
            Row row = sheet.createRow(startRowIndex++);
            createCellCount(row, 0, data, cellStyle);
        }
    }

    private void createCellCount(Row row, int CellStartIndex, String[] data, CellStyle cellStyle) {
        for (int i = CellStartIndex, y = 0; y < data.length; i++, y++) {
            Cell cell = row.createCell(i);
            cell.setCellStyle(cellStyle);
            cell.setCellValue(data[y]);

        }
    }

    private List<String[]> getStaffStatisticsData(List<StaffStatisticsBo> staffStatisticsBos) {
        List<String[]> data = new ArrayList<>(staffStatisticsBos.size());
        for (int i = 0; i < staffStatisticsBos.size(); i++) {
            String[] staffData = getData(i + 1, staffStatisticsBos.get(i));
            data.add(staffData);
        }
        return data;
    }

    private String[] getData(int index, StaffStatisticsBo staffStatisticsBo) {
        String companyNames = ArrayUtil.join(staffStatisticsBo.getCompanyNames().toArray(), ",");
        String postinNames = ArrayUtil.join(staffStatisticsBo.getPostionNames().toArray(), ",");
        String sellAmount = staffStatisticsBo.getSellAmount().toString();
        String royaltyAmount = staffStatisticsBo.getRoyaltyAmount().toString();
        String member = String.format("%s/%s", staffStatisticsBo.getMemberOrderNum(), staffStatisticsBo.getMemberAmount());
        String rentVehicle = String.format("%s/%s", staffStatisticsBo.getRentVehicleOrderNum(), staffStatisticsBo.getRentVehicleAmount());
        String travel = String.format("%s/%s", staffStatisticsBo.getTravelOrderNum(), staffStatisticsBo.getTravelAmount());
        return new String[]{String.valueOf(index), staffStatisticsBo.getName(), companyNames, postinNames, staffStatisticsBo.getPhone(),sellAmount, royaltyAmount, member, rentVehicle, travel};
    }

    private void createHeader(Row row, int cellStartIndex, String[] header, CellStyle cellStyle) {
        for (int i = 0; i < header.length; i++) {
            Cell cell = row.createCell(cellStartIndex);
            cell.setCellValue(header[i]);
            cell.setCellStyle(cellStyle);
            cellStartIndex += 1;
        }
    }

    private CellStyle createGeneralCellStyle(XSSFWorkbook hssfWorkbook) {
        CellStyle cellStyleGeneral = createHeaderCellStyle(hssfWorkbook);
        cellStyleGeneral.setWrapText(true);
        XSSFFont generalFont = createFont(hssfWorkbook);
        generalFont.setBold(false);
        cellStyleGeneral.setFont(generalFont);
        return cellStyleGeneral;
    }

    private CellStyle createHeaderCellStyle(XSSFWorkbook hssfWorkbook) {
        XSSFCellStyle cellStyleHeader = hssfWorkbook.createCellStyle();
        cellStyleHeader.setAlignment(HorizontalAlignment.CENTER);
        cellStyleHeader.setVerticalAlignment(VerticalAlignment.CENTER);
        cellStyleHeader.setLocked(false);
        XSSFFont headerFont = createFont(hssfWorkbook);
        cellStyleHeader.setFont(headerFont);
        return cellStyleHeader;
    }

    private XSSFFont createFont(XSSFWorkbook hssfWorkbook) {
        XSSFFont font = hssfWorkbook.createFont();
        font.setFontName("黑体");
        font.setBold(true);
        return font;
    }

    /**
     * 员工业绩统计
     *
     * @param staffStatisticsFindDTO
     * @return
     */
    public Map<String, Object> listStaffStatisticsWithPage(StaffStatisticsFindDTO staffStatisticsFindDTO) {
        Map<String, Object> result = new HashMap<>(2);
        PageDataVO<StaffStatisticsBo> pageDataVO = new PageDataVO<>();

        //1.分页查询全部员工
        PageDataVO<UserStaffBo> staffPage = appUserDetailBiz.findAllStaffsByCompanyIdAndPostionIdWithPage(staffStatisticsFindDTO);
        List<UserStaffBo> allStaffs = staffPage.getData();
        if (CollectionUtils.isEmpty(allStaffs)) {
            result.put(TOTAL_VAL, new StaffStatisticsTotalBo());
            pageDataVO.setData(Collections.EMPTY_LIST);
            pageDataVO.setPageSize(staffStatisticsFindDTO.getLimit());
            pageDataVO.setPageNum(staffStatisticsFindDTO.getPage());
            pageDataVO.setTotalPage(0);
            pageDataVO.setTotalCount(0L);
            result.put(PAGE_VAL, pageDataVO);
            return result;
        }
        List<StaffStatisticsBo> staffStatisticsBos = processStaffStatistics(allStaffs, staffStatisticsFindDTO);
        if (staffStatisticsFindDTO.getPage() == 1) {
            StaffStatisticsTotalBo staffStatisticsTotalBo = getStaffStatisticsTotalBo(allStaffs.size(), staffStatisticsBos);
            result.put(TOTAL_VAL, staffStatisticsTotalBo);
        }
        pageDataVO.setData(staffStatisticsBos);
        pageDataVO.setPageNum(staffPage.getPageNum());
        pageDataVO.setPageSize(staffPage.getPageSize());
        pageDataVO.setTotalCount(staffPage.getTotalCount());
        pageDataVO.setTotalPage(staffPage.getTotalPage());
        result.put(PAGE_VAL, pageDataVO);
        return result;
    }

    private StaffStatisticsTotalBo getStaffStatisticsTotalBo(Integer size, List<StaffStatisticsBo> staffStatisticsBos) {
        StaffStatisticsTotalBo staffStatisticsTotalBo = new StaffStatisticsTotalBo();
        Supplier<Stream<StaffStatisticsBo>> streamSupplier = () -> staffStatisticsBos.stream();
        BigDecimal royaltyAmount = streamSupplier.get().map(StaffStatisticsBo::getRoyaltyAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
        BigDecimal sellAmount = streamSupplier.get().map(StaffStatisticsBo::getSellAmount).reduce(BigDecimal.ZERO, BigDecimal::add);
        staffStatisticsTotalBo.setTotalRoyaltyAmount(royaltyAmount);
        staffStatisticsTotalBo.setTotalSellAmount(sellAmount);
        staffStatisticsTotalBo.setStaffNum(size);
        return staffStatisticsTotalBo;
    }

    private List<StaffStatisticsBo> processStaffStatistics(List<UserStaffBo> allStaffs, StaffStatisticsFindDTO staffStatisticsFindDTO) {
        List<StaffStatisticsBo> staffStatisticsBos = new ArrayList<>();
        //2.根据员工id范围和其他条件分页查询
        List<Integer> userIds = allStaffs.stream().map(UserStaffBo::getUserId).collect(Collectors.toList());
        List<AppUserSellingWaterDTO> appUserSellingWaterDTOS = appUserSellingWaterBiz.findStatffSellerWaterByUserIdsAndTime(userIds, staffStatisticsFindDTO.getStartDate(), staffStatisticsFindDTO.getEndDate());
        Map<Integer, Map<String, Map<Integer, List<AppUserSellingWaterDTO>>>> sellingWaterMap = new HashMap<>();

        if (CollectionUtils.isNotEmpty(appUserSellingWaterDTOS)) {
            //3.根据订单id查询订单相关信息
            List<Integer> orderIds = appUserSellingWaterDTOS.stream().map(AppUserSellingWaterDTO::getOrderId).collect(Collectors.toList());
            ObjectRestResponse<List<OrderDTO>> response = orderFeign.findOrdersByorderId(orderIds);
            List<OrderDTO> orders = response.getData();
            Map<Integer, Integer> ordersMap = orders.stream().collect(Collectors.toMap(OrderDTO::getId, OrderDTO::getCompanyId));
            for (AppUserSellingWaterDTO appUserSellingWaterDTO : appUserSellingWaterDTOS) {
                appUserSellingWaterDTO.setCompanyId(ordersMap.get(appUserSellingWaterDTO.getOrderId()) == null ? NO_COMPANY_STATE : ordersMap.get(appUserSellingWaterDTO.getOrderId()));
                appUserSellingWaterDTO.setStateGroup(String.format("%d-%d-%d", appUserSellingWaterDTO.getUserId(), appUserSellingWaterDTO.getPositionId(), appUserSellingWaterDTO.getCompanyId()));
            }


            sellingWaterMap = appUserSellingWaterDTOS.stream().collect(Collectors.groupingBy(AppUserSellingWaterDTO::getUserId,
                    Collectors.groupingBy(AppUserSellingWaterDTO::getStateGroup, Collectors.groupingBy(AppUserSellingWaterDTO::getOrderType,Collectors.toList()))));
        }

        Map<Integer, String> companyMap = vehicleFeign.findCompanyMap();
        List<WalletPostionVo> allPostions = appUserPositionBiz.findAllPostions();
        Map<Integer, String> postionMap = allPostions.stream().collect(Collectors.toMap(WalletPostionVo::getId, WalletPostionVo::getName));
        for (UserStaffBo userStaffBo : allStaffs) {
            String stateGroup = String.format("%d-%d-%d", userStaffBo.getUserId(), userStaffBo.getPostionId(), userStaffBo.getCompanyId() == null ? NO_COMPANY_STATE : userStaffBo.getCompanyId());
            StaffStatisticsBo staffStatisticsBo = creatDefaultStaffStatistics(userStaffBo);
            if (!sellingWaterMap.isEmpty()) {
                Map<String, Map<Integer, List<AppUserSellingWaterDTO>>> userSellingWaterMap = sellingWaterMap.get(userStaffBo.getUserId());
                if (userSellingWaterMap != null && !userSellingWaterMap.isEmpty()) {
                    AtomicReference<BigDecimal> royaltyAmountAtomicReference = new AtomicReference<>(BigDecimal.ZERO);
                    List<String> postionNames = Lists.newArrayList(staffStatisticsBo.getPostionNames());
                    List<Integer> postionIds = Lists.newArrayList(staffStatisticsBo.getPostionIds());
                    List<String> companyNames = Lists.newArrayList(staffStatisticsBo.getCompanyNames());
                    List<Integer> compnayIds = Lists.newArrayList(staffStatisticsBo.getCompnayIds());
                    userSellingWaterMap.forEach((K, V) -> {
                        if (!K.equals(stateGroup)) {
                            String[] sates = K.split("-");
                            String postionIdStr = sates[1];
                            postionMap.get(Integer.valueOf(postionIdStr));
                            String companyIdStr = sates[2];
                            if (!postionIds.contains(Integer.valueOf(postionIdStr))) {
                                postionNames.add(postionMap.get(Integer.valueOf(postionIdStr)));
                                postionIds.add(Integer.valueOf(postionIdStr));
                            }

                            if (!compnayIds.contains(Integer.valueOf(companyIdStr))) {
                                String companyName = companyMap.get(Integer.valueOf(companyIdStr));
                                if (StringUtils.hasText(companyName)) {
                                    companyNames.add(companyName);
                                    compnayIds.add(Integer.valueOf(companyIdStr));
                                }
                            }
                            staffStatisticsBo.setPostionIds(postionIds);
                            staffStatisticsBo.setPostionNames(postionNames);
                            staffStatisticsBo.setCompnayIds(compnayIds);
                            staffStatisticsBo.setCompanyNames(companyNames);
                        }
                        wrapStaffStatistics(V, royaltyAmountAtomicReference, staffStatisticsBo);
                    });
                    staffStatisticsBo.setCompnayIds(compnayIds);
                    staffStatisticsBo.setCompanyNames(companyNames);
                    staffStatisticsBo.setPostionIds(postionIds);
                    staffStatisticsBo.setPostionNames(postionNames);
                    staffStatisticsBo.setRoyaltyAmount(royaltyAmountAtomicReference.get());
                }
            }
            staffStatisticsBos.add(staffStatisticsBo);
        }
        return staffStatisticsBos;
    }

    private StaffStatisticsBo wrapStaffStatistics(Map<Integer, List<AppUserSellingWaterDTO>> waterDTOSMap, AtomicReference<BigDecimal> royaltyAmountAtomicReference, StaffStatisticsBo staffStatisticsBo) {
        waterDTOSMap.forEach((K, V) -> {
            Map<String, Object> orderRentResultMap = wrapStaffStatisticsMap(V);
            BigDecimal royaltyAmount = royaltyAmountAtomicReference.get();
            royaltyAmount = royaltyAmount.add((BigDecimal) orderRentResultMap.get(TOTAL_COMMISSION_VAL));
            royaltyAmountAtomicReference.set(royaltyAmount);
            switch (K) {
                case StatisticsStatusEnum.ORDER_RENT_VEHICLE_TYPE:
                    staffStatisticsBo.setRentVehicleOrderNum(((Integer) orderRentResultMap.get(TOTAL_NUM_VAL)) + staffStatisticsBo.getRentVehicleOrderNum());
                    staffStatisticsBo.setRentVehicleAmount(((BigDecimal) orderRentResultMap.get(TOTAL_AMOUNT_VAL)).add(staffStatisticsBo.getRentVehicleAmount()));
                    break;
                case StatisticsStatusEnum.ORDER_TOUR_TYPE:
                    staffStatisticsBo.setTravelOrderNum(((Integer) orderRentResultMap.get(TOTAL_NUM_VAL)) + staffStatisticsBo.getTravelOrderNum());
                    staffStatisticsBo.setTravelAmount(((BigDecimal) orderRentResultMap.get(TOTAL_AMOUNT_VAL)).add(staffStatisticsBo.getTravelAmount()));
                    break;
                case StatisticsStatusEnum.ORDER_MEMBER_TYPE:
                    staffStatisticsBo.setMemberOrderNum(((Integer) orderRentResultMap.get(TOTAL_NUM_VAL)) + staffStatisticsBo.getMemberOrderNum());
                    staffStatisticsBo.setMemberAmount(((BigDecimal) orderRentResultMap.get(TOTAL_AMOUNT_VAL)).add(staffStatisticsBo.getMemberAmount()));
                    break;
                default:
                    break;
            }
        });
        return staffStatisticsBo;
    }

    private Map<String, Object> wrapStaffStatisticsMap(List<AppUserSellingWaterDTO> sellingWaters) {
        Map<String, Object> staffStatisticsResultMap = new HashMap<>(2);
        AtomicInteger orderCounter = new AtomicInteger(0);
        if (CollectionUtils.isEmpty(sellingWaters)) {
            staffStatisticsResultMap.put(TOTAL_AMOUNT_VAL, BigDecimal.ZERO);
            staffStatisticsResultMap.put(TOTAL_NUM_VAL, orderCounter.get());
        }
        BigDecimal totalAmount = BigDecimal.ZERO;
        BigDecimal royaltyAmount = BigDecimal.ZERO;
            Map<Integer, List<AppUserSellingWaterDTO>> orderSellingMap = sellingWaters.stream().collect(Collectors.groupingBy(AppUserSellingWaterDTO::getOrderId, Collectors.toList()));
            Set<Map.Entry<Integer, List<AppUserSellingWaterDTO>>> orderSellingSet = orderSellingMap.entrySet();
            for (Map.Entry<Integer, List<AppUserSellingWaterDTO>> orderSellingEntry : orderSellingSet) {
                BigDecimal inAmount = BigDecimal.ZERO;
                BigDecimal outAmount = BigDecimal.ZERO;
                boolean isCommissionZero=false;
                List<AppUserSellingWaterDTO> appUserSellingWaterDTOS = orderSellingEntry.getValue();
                for (AppUserSellingWaterDTO waterDTO : appUserSellingWaterDTOS) {
                    if (waterDTO.getStatus() == 0) {
                        royaltyAmount = royaltyAmount.add(waterDTO.getCommission());
                        totalAmount = totalAmount.add(waterDTO.getCommission().divide(new BigDecimal(String.valueOf(waterDTO.getExtract()/100.00)),2,BigDecimal.ROUND_HALF_UP));
                        inAmount = inAmount.add(waterDTO.getCommission());
                        isCommissionZero = waterDTO.getCommission().intValue()==0;
                    }

                    if (waterDTO.getStatus() == 1) {
                        royaltyAmount = royaltyAmount.subtract(waterDTO.getCommission());
                        totalAmount = totalAmount.subtract(waterDTO.getCommission().divide(new BigDecimal(String.valueOf(waterDTO.getExtract()/100.00)),2,BigDecimal.ROUND_HALF_UP));
                        outAmount = outAmount.add(waterDTO.getCommission());
                        isCommissionZero=!isCommissionZero;
                    }
                }

                if(appUserSellingWaterDTOS.size()==1 || inAmount.subtract(outAmount).doubleValue()>0 || isCommissionZero){
                    orderCounter.incrementAndGet();
                }
            }
            staffStatisticsResultMap.put(TOTAL_AMOUNT_VAL, totalAmount);
            staffStatisticsResultMap.put(TOTAL_NUM_VAL, orderCounter.get());
            staffStatisticsResultMap.put(TOTAL_COMMISSION_VAL, royaltyAmount);
        return staffStatisticsResultMap;
    }

    private StaffStatisticsBo creatDefaultStaffStatistics(UserStaffBo userStaffBo) {
        StaffStatisticsBo staffStatisticsBo = new StaffStatisticsBo();
        if (userStaffBo.getCompanyId() != null) {
            staffStatisticsBo.setCompanyNames(Arrays.asList(userStaffBo.getCompanyName()));
            staffStatisticsBo.setCompnayIds(Arrays.asList(userStaffBo.getCompanyId()));
        }
        if (userStaffBo.getPostionId() != null) {
            staffStatisticsBo.setPostionNames(Arrays.asList(userStaffBo.getPostionName()));
            staffStatisticsBo.setPostionIds(Arrays.asList(userStaffBo.getPostionId()));
        }
        staffStatisticsBo.setUserId(userStaffBo.getUserId());
        staffStatisticsBo.setName(StringUtils.hasText(userStaffBo.getRealName()) ? userStaffBo.getRealName() : userStaffBo.getNickName());
        staffStatisticsBo.setPhone(userStaffBo.getPhone());
        staffStatisticsBo.setMemberAmount(BigDecimal.ZERO);
        staffStatisticsBo.setMemberOrderNum(0);
        staffStatisticsBo.setTravelAmount(BigDecimal.ZERO);
        staffStatisticsBo.setTravelOrderNum(0);
        staffStatisticsBo.setRentVehicleAmount(BigDecimal.ZERO);
        staffStatisticsBo.setRentVehicleOrderNum(0);
        staffStatisticsBo.setSellAmount(BigDecimal.ZERO);
        staffStatisticsBo.setRoyaltyAmount(BigDecimal.ZERO);
        return staffStatisticsBo;
    }

}