package com.xxfc.platform.order.biz;

import cn.hutool.core.util.ObjectUtil;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.google.common.collect.Lists;
import com.xxfc.platform.order.contant.enumerate.StatisticsStatusEnum;
import com.xxfc.platform.order.entity.OrderRentVehicleReceivedStatistics;
import com.xxfc.platform.order.mapper.OrderRentVehicleReceivedStatisticsMapper;
import com.xxfc.platform.order.pojo.account.OrderAccountBo;
import com.xxfc.platform.order.pojo.account.OrderAccountDetail;
import com.xxfc.platform.order.pojo.dto.OrderDTO;
import com.xxfc.platform.order.pojo.dto.OrderReceivedStatisticsFindDTO;
import com.xxfc.platform.vehicle.feign.VehicleFeign;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 租车订单统计
 *
 * @author libin
 * @email 18178966185@163.com
 * @date 2019-11-08 18:03:42
 */
@Service
@Transactional(rollbackFor = Exception.class)
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
public class OrderRentVehicleReceivedStatisticsBiz extends BaseBiz<OrderRentVehicleReceivedStatisticsMapper, OrderRentVehicleReceivedStatistics> {

    private final BaseOrderBiz baseOrderBiz;
    private final OrderAccountBiz orderAccountBiz;
    private final VehicleFeign vehicleFeign;
    private final String NO_PAY_STATE_STR = String.format("%s-%d", "-", StatisticsStatusEnum.NO_PAY_WAY);
    private final int PAY_ORDER = 101;

    /**
     * 租车统计查询
     * @param orderReceivedStatisticsFindDTO
     * @return
     */
    public List<OrderRentVehicleReceivedStatistics> selectOrderReceivedStatistics(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO) {
        return mapper.selectOrderRentVehicleReceivedStatistics(orderReceivedStatisticsFindDTO);
    }

    /**
     * 租车统计处理
     * @param startDate 开始时间
     * @param endDate 结束时间
     */
    public void orderRentVehicleReceivedStatistics(Date startDate, Date endDate) {
        List<OrderRentVehicleReceivedStatistics> orderRentVehicleReceivedStatisticsList = new ArrayList<>();

        //根据租车订单类型 时间查询
        List<OrderDTO> orderDTOS = baseOrderBiz.selectOrdersByTypeAndTime(Arrays.asList(StatisticsStatusEnum.ORDER_RENT_VEHICLE_TYPE), startDate, endDate);
        //数据处理 状态组合
        Map<String, Long> stateGroupMap = orderDTOS.stream().peek(x -> {
            x.setStateGroup(String.format("%d-%d-%d-%d", x.getCompanyId(), x.getOrderOrigin(), x.getPayWay() == null ? StatisticsStatusEnum.NO_PAY_WAY : x.getPayWay(), x.getStatus()));
        })
                .collect(Collectors.groupingBy(OrderDTO::getStateGroup, Collectors.counting()));

        //订单账目信息
        List<OrderAccountBo> orderAccountBoList = orderAccountBiz.selectByDate(startDate, endDate);
        //账目数据处理 状态组合
        Map<String, List<OrderAccountBo>> ordersMap = orderAccountBoList.stream().peek(x -> {
            x.setStateGroup(String.format("%d-%d-%d-%d", x.getCompanyId(), x.getOrderOrigin(), x.getPayWay() == null ? StatisticsStatusEnum.NO_PAY_WAY : x.getPayWay(), x.getStatus()));
        }).collect(Collectors.groupingBy(OrderAccountBo::getStateGroup, Collectors.toList()));

        Set<Map.Entry<String, List<OrderAccountBo>>> ordersSet = ordersMap.entrySet();
        for (Map.Entry<String, List<OrderAccountBo>> orderEntry : ordersSet) {
            String orderKey = orderEntry.getKey();
            List<OrderAccountBo> orderAccountBos = orderEntry.getValue();
            OrderRentVehicleReceivedStatistics orderRentVehicleReceivedStatistics = StatisticsStatusEnum.wrapStatisticsObject(startDate, orderKey, new OrderRentVehicleReceivedStatistics());
            BigDecimal totalAmount = BigDecimal.ZERO;
            BigDecimal refundAmount = BigDecimal.ZERO;
            for (OrderAccountBo orderAccountBo : orderAccountBos) {
                OrderAccountDetail accountDetailEntity = orderAccountBo.getAccountDetailEntity();
                if (orderAccountBo.getAccountType() == PAY_ORDER) {
                    totalAmount.add(accountDetailEntity.getOrderAmount()).add(accountDetailEntity.getDepositAmount());
                } else {
                    refundAmount.add(accountDetailEntity.getOrderAmount()).add(accountDetailEntity.getDepositAmount());
                }
            }
            orderRentVehicleReceivedStatistics.setTotalAmount(totalAmount.subtract(refundAmount));
            Long totalQuantity = stateGroupMap == null ? 0L : stateGroupMap.get(orderKey) == null ? 0L : stateGroupMap.get(orderKey);
            orderRentVehicleReceivedStatistics.setTotalQuantity(totalQuantity.intValue());
        }

        //未支付单
        List<BigDecimal> noPayOrdersPrices = orderDTOS.stream().filter(x -> x.getPayWay() != null).map(OrderDTO::getRealAmount).collect(Collectors.toList());
        BigDecimal totalNoPayAmount = CollectionUtils.isEmpty(noPayOrdersPrices) ? BigDecimal.ZERO : noPayOrdersPrices.stream().reduce(BigDecimal.ZERO, (x, y) -> x.add(y));
        int totalNoPayOrderQuantity = CollectionUtils.isEmpty(noPayOrdersPrices) ? 0 : noPayOrdersPrices.size();

        //查询分公司ids
        List<Integer> companyIds = vehicleFeign.findCompanyIdsByAreaId(null);
        //创建剩余状态组合的租车统计对象
        List stisticsActiveState = ordersMap.isEmpty() ? Collections.EMPTY_LIST : Lists.newArrayList(ordersMap.keySet());
        List<OrderRentVehicleReceivedStatistics> otherStatisticsStateGroupList = createOtherStatisticsStateGroupList(startDate, stisticsActiveState, companyIds, totalNoPayAmount, totalNoPayOrderQuantity);
        orderRentVehicleReceivedStatisticsList.addAll(otherStatisticsStateGroupList);
        //保存
        insertMemberReceivedStatisticsBatch(orderRentVehicleReceivedStatisticsList);
    }

    /**
     * 创建剩余状态数据
     *
     * @param startDate  时间
     * @param statisticsStateGroups 状态组合 集合
     * @param companyIds  公司ids
     * @param totalNoPayAmount  未支付金额
     * @param totalNoPayOrderQuantity  未支付订单数
     * @return
     */
    private List<OrderRentVehicleReceivedStatistics> createOtherStatisticsStateGroupList(Date startDate,
                                                                                         List<String> statisticsStateGroups,
                                                                                         List<Integer> companyIds,
                                                                                         BigDecimal totalNoPayAmount,
                                                                                         Integer totalNoPayOrderQuantity) {

        List<OrderRentVehicleReceivedStatistics> orderRentVehicleReceivedStatisticsList = new ArrayList<>();
         //获取剩余状态组合
        List<String> otherStatisticsStateGroup = StatisticsStatusEnum.getOtherStatisticsStateGroup(companyIds, statisticsStateGroups);
        //创建租车统计克隆对象
        OrderRentVehicleReceivedStatistics orderRentVehicleReceivedStatistics = new OrderRentVehicleReceivedStatistics();
        //统计对象的生成
        otherStatisticsStateGroup.parallelStream().map(stateGroup -> {
            OrderRentVehicleReceivedStatistics orderRentVehicleReceivedStatisticsClone = StatisticsStatusEnum.wrapStatisticsObject(startDate, stateGroup,ObjectUtil.cloneByStream(orderRentVehicleReceivedStatistics));
            orderRentVehicleReceivedStatisticsClone = wrapOrderRentVehicleReceivedStatistics(stateGroup,totalNoPayAmount,totalNoPayOrderQuantity,orderRentVehicleReceivedStatisticsClone);
            orderRentVehicleReceivedStatisticsList.add(orderRentVehicleReceivedStatisticsClone);
            return orderRentVehicleReceivedStatisticsClone;
        }).count();
        return orderRentVehicleReceivedStatisticsList;
    }

    /**
     *
     * @param stateGroup  状态组合
     * @param totalNoPayAmount 未支付金额
     * @param totalNoPayOrderQuantity  未支付订单量
     * @param orderRentVehicleReceivedStatistics 租车统计对象
     * @return
     */
    private OrderRentVehicleReceivedStatistics wrapOrderRentVehicleReceivedStatistics(String stateGroup,BigDecimal totalNoPayAmount, Integer totalNoPayOrderQuantity,OrderRentVehicleReceivedStatistics orderRentVehicleReceivedStatistics){
        if (stateGroup.contains(NO_PAY_STATE_STR)) {
            orderRentVehicleReceivedStatistics.setTotalAmount(totalNoPayAmount);
            orderRentVehicleReceivedStatistics.setTotalQuantity(totalNoPayOrderQuantity);
        } else {
            orderRentVehicleReceivedStatistics.setTotalAmount(BigDecimal.ZERO);
            orderRentVehicleReceivedStatistics.setTotalQuantity(0);
        }
        return orderRentVehicleReceivedStatistics;
    }

    /**
     * 批量插入数据
     * @param orderRentVehicleReceivedStatistics
     */
    public void insertMemberReceivedStatisticsBatch(List<OrderRentVehicleReceivedStatistics> orderRentVehicleReceivedStatistics) {
        mapper.insertList(orderRentVehicleReceivedStatistics);
    }
}