package com.xxfc.platform.order.biz;

import cn.hutool.core.util.ObjectUtil;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.github.wxiaoqi.security.common.enumconstant.LevelEnum;
import com.google.common.collect.Lists;
import com.xxfc.platform.order.contant.enumerate.StatisticsStatusEnum;
import com.xxfc.platform.order.entity.OrderMemberReceivedStatistics;
import com.xxfc.platform.order.mapper.OrderMemberReceivedStatisticsMapper;
import com.xxfc.platform.order.pojo.dto.OrderDTO;
import com.xxfc.platform.order.pojo.dto.OrderReceivedStatisticsFindDTO;
import lombok.RequiredArgsConstructor;
import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 会员订单统计
 *
 * @author libin
 * @email 18178966185@163.com
 * @date 2019-11-08 18:03:42
 */
@Service
@Transactional(rollbackFor = Exception.class)
@RequiredArgsConstructor(onConstructor = @__(@Autowired))
public class OrderMemberReceivedStatisticsBiz extends BaseBiz<OrderMemberReceivedStatisticsMapper, OrderMemberReceivedStatistics> {
    private final BaseOrderBiz baseOrderBiz;

    /**
     * 会员统计查询
     *
     * @param orderReceivedStatisticsFindDTO
     * @return
     */
    public List<OrderMemberReceivedStatistics> selectOrderReceivedStatistics(OrderReceivedStatisticsFindDTO orderReceivedStatisticsFindDTO) {
        return mapper.selectOrderMemberReceivedStatistics(orderReceivedStatisticsFindDTO);
    }

    /**
     * 会员统计处理
     *
     * @param startDate  开始时间
     * @param endDate  结束时间
     */
    public void orderMemberReceivedStatistics(Date startDate, Date endDate) {
        List<OrderMemberReceivedStatistics> orderMemberReceivedStatisticsList = new ArrayList<>(18);

        //根据时间范围查询出会员单
        List<OrderDTO> orders = baseOrderBiz.selectOrdersByTypeAndTime(Arrays.asList(StatisticsStatusEnum.ORDER_TOUR_TYPE), startDate, endDate);

        //数据处理 把状态组合
        Map<String, Map<LevelEnum, List<OrderDTO>>> ordersMap = orders.stream().peek(x -> {
            x.setStateGroup(String.format("%d-%d-%d-%d", StatisticsStatusEnum.DEFAULT_COMPANY, x.getOrderOrigin(), x.getPayWay() == null ? StatisticsStatusEnum.NO_PAY_WAY : x.getPayWay(), x.getHasPay()));
        }).collect(Collectors.groupingBy(OrderDTO::getStateGroup, Collectors.groupingBy(OrderDTO::getLevelEnum, Collectors.toList())));

        if (!ordersMap.isEmpty()) {
            Set<Map.Entry<String, Map<LevelEnum, List<OrderDTO>>>> orderSet = ordersMap.entrySet();
            for (Map.Entry<String, Map<LevelEnum, List<OrderDTO>>> orderEntry : orderSet) {
                //获取状态
                String orderStatisticsGroup = orderEntry.getKey();
                Map<LevelEnum, List<OrderDTO>> orderEntryValue = orderEntry.getValue();
                //创建会员统计对象
                OrderMemberReceivedStatistics orderMemberReceivedStatistics = StatisticsStatusEnum.wrapStatisticsObject(startDate, orderStatisticsGroup, new OrderMemberReceivedStatistics());
                //等级金额 和量 的计算
                Set<Map.Entry<LevelEnum, List<OrderDTO>>> memberOrdersSet = orderEntryValue.entrySet();
                for (Map.Entry<LevelEnum, List<OrderDTO>> levelEnumListEntry : memberOrdersSet) {
                    orderMemberReceivedStatistics = wrapOrderMemberReceivedStatistics(levelEnumListEntry.getValue(), levelEnumListEntry.getKey(), orderMemberReceivedStatistics);
                }
                //总金额
                BigDecimal totalAmount = orderMemberReceivedStatistics.getToalCommonAmmount()
                        .add(orderMemberReceivedStatistics.getTotalGoldAmount())
                        .add(orderMemberReceivedStatistics.getTotalDiamondAmmount());

                //总量
                Integer totalQuantity = orderMemberReceivedStatistics.getTotalCommonQuantity() +
                        orderMemberReceivedStatistics.getTotalGoldQuantity() +
                        orderMemberReceivedStatistics.getTotalDiamondQuantity();

                orderMemberReceivedStatistics.setTotalQuantity(totalQuantity);
                orderMemberReceivedStatistics.setTotalAmount(totalAmount);

                orderMemberReceivedStatisticsList.add(orderMemberReceivedStatistics);
            }
        }
        //创建剩余状态数据
        List stisticsActiveState = ordersMap.isEmpty() ? Collections.EMPTY_LIST : Lists.newArrayList(ordersMap.keySet());
        List<OrderMemberReceivedStatistics> otherStatisticsStateGroupList = createOtherStatisticsStateGroupList(startDate, stisticsActiveState, StatisticsStatusEnum.DEFAULT_COMPANY);
        orderMemberReceivedStatisticsList.addAll(otherStatisticsStateGroupList);
        //保存
        insertMemberReceivedStatisticsBatch(orderMemberReceivedStatisticsList);
    }

    /**
     * @param orders 某种状态下的会员单
     * @param levelEnum 会员等级
     * @param orderMemberReceivedStatistics 会员统计对象
     * @return
     */
    private OrderMemberReceivedStatistics wrapOrderMemberReceivedStatistics(List<OrderDTO> orders, LevelEnum levelEnum, OrderMemberReceivedStatistics orderMemberReceivedStatistics) {
        orders = CollectionUtils.isEmpty(orders) ? Collections.EMPTY_LIST : orders;
        BigDecimal totalAmount = orders.stream().map(OrderDTO::getRealAmount).reduce(BigDecimal.ZERO, (x, y) -> x.add(y));
        switch (levelEnum) {
            case GENERAL:
                orderMemberReceivedStatistics.setToalCommonAmmount(totalAmount);
                orderMemberReceivedStatistics.setTotalCommonQuantity(orders.size());
                break;
            case GOLD:
                orderMemberReceivedStatistics.setTotalGoldAmount(totalAmount);
                orderMemberReceivedStatistics.setTotalGoldQuantity(orders.size());
                break;
            case DIAMOND:
                orderMemberReceivedStatistics.setTotalDiamondAmmount(totalAmount);
                orderMemberReceivedStatistics.setTotalDiamondQuantity(orders.size());
                break;
            default:
                break;
        }
        return orderMemberReceivedStatistics;
    }

    /**
     * 创建剩余状态数据
     *
     * @param startDate  时间
     * @param statisticsStateGroups 状态组合 集合
     * @param companyId  公司 id
     * @return
     */
    private List<OrderMemberReceivedStatistics> createOtherStatisticsStateGroupList(Date startDate, List<String> statisticsStateGroups, Integer companyId) {
        List<OrderMemberReceivedStatistics> orderMemberReceivedStatisticsList = new ArrayList<>();
        //获取剩余状态组合
        List<String> otherStatisticsStateGroup = StatisticsStatusEnum.getOtherStatisticsStateGroup(Arrays.asList(companyId), statisticsStateGroups);
        //创建会员克隆统计对象
        OrderMemberReceivedStatistics orderMemberReceivedStatistics = createDefaultOrderMemberReceivedStatistics();
        //统计对象的生成
        otherStatisticsStateGroup.parallelStream().peek(stateGroup -> {
            OrderMemberReceivedStatistics orderMemberReceivedStatisticsClone = StatisticsStatusEnum.wrapStatisticsObject(startDate, stateGroup, ObjectUtil.cloneByStream(orderMemberReceivedStatistics));
            orderMemberReceivedStatisticsList.add(orderMemberReceivedStatisticsClone);
        }).count();
        return orderMemberReceivedStatisticsList;
    }

    /**
     * 创建默认的会员订单统计对象
     *
     * @return
     */
    private OrderMemberReceivedStatistics createDefaultOrderMemberReceivedStatistics() {
        OrderMemberReceivedStatistics orderMemberReceivedStatistics = new OrderMemberReceivedStatistics();
        orderMemberReceivedStatistics.setTotalAmount(BigDecimal.ZERO);
        orderMemberReceivedStatistics.setTotalCommonQuantity(0);
        orderMemberReceivedStatistics.setToalCommonAmmount(BigDecimal.ZERO);
        orderMemberReceivedStatistics.setTotalGoldQuantity(0);
        orderMemberReceivedStatistics.setTotalGoldAmount(BigDecimal.ZERO);
        orderMemberReceivedStatistics.setTotalDiamondQuantity(0);
        orderMemberReceivedStatistics.setTotalDiamondAmmount(BigDecimal.ZERO);
        return orderMemberReceivedStatistics;
    }

    /**
     * 批量插入
     *
     * @param orderMemberReceivedStatistics
     */
    public void insertMemberReceivedStatisticsBatch(List<OrderMemberReceivedStatistics> orderMemberReceivedStatistics) {
        mapper.insertList(orderMemberReceivedStatistics);
    }
}