package com.xxfc.platform.order.rest;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.util.StrUtil;
import cn.hutool.json.JSONUtil;
import com.github.wxiaoqi.security.admin.feign.UserFeign;
import com.github.wxiaoqi.security.admin.feign.dto.UserDTO;
import com.github.wxiaoqi.security.admin.vo.AppUserVo;
import com.github.wxiaoqi.security.auth.client.annotation.IgnoreClientToken;
import com.github.wxiaoqi.security.auth.client.annotation.IgnoreUserToken;
import com.github.wxiaoqi.security.common.context.BaseContextHandler;
import com.github.wxiaoqi.security.common.exception.BaseException;
import com.github.wxiaoqi.security.common.msg.ObjectRestResponse;
import com.github.wxiaoqi.security.common.rest.CommonBaseController;
import com.github.wxiaoqi.security.common.util.ClientUtil;
import com.github.wxiaoqi.security.common.util.Query;
import com.github.wxiaoqi.security.common.util.process.ResultCode;
import com.github.wxiaoqi.security.common.vo.PageDataVO;
import com.github.wxiaoqi.security.common.vo.PageParam;
import com.xxfc.platform.order.biz.BaseOrderBiz;
import com.xxfc.platform.order.biz.OrderMemberDetailBiz;
import com.xxfc.platform.order.contant.enumerate.OrderStatusEnum;
import com.xxfc.platform.order.entity.BaseOrder;
import com.xxfc.platform.order.mqhandler.RabbitProduct;
import com.xxfc.platform.order.pojo.order.CancelOrderDTO;
import com.xxfc.platform.order.pojo.order.OrderPageVO;
import com.xxfc.platform.order.pojo.order.UnitPriceDTO;
import com.xxfc.platform.order.pojo.pay.RentVehicleOrderPayVO;
import com.xxfc.platform.universal.constant.DictionaryKey;
import com.xxfc.platform.universal.constant.enumerate.PayChannelEnum;
import com.xxfc.platform.universal.entity.Dictionary;
import com.xxfc.platform.universal.feign.ThirdFeign;
import com.xxfc.platform.universal.vo.OrderPayVo;
import com.xxfc.platform.vehicle.common.RestResponse;
import com.xxfc.platform.vehicle.entity.Vehicle;
import com.xxfc.platform.vehicle.feign.VehicleFeign;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiModelProperty;
import io.swagger.annotations.ApiOperation;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static com.xxfc.platform.universal.constant.DictionaryKey.APP_ORDER;

@Controller
@RequestMapping("baseOrder")
@Api(value = "订单", tags = {"订单"})
@Slf4j
public class BaseOrderController extends CommonBaseController {

    public static final Integer STATUS_SUCCESS = 1;
    @Autowired
    BaseOrderBiz baseOrderBiz;
    @Autowired
    ThirdFeign thirdFeign;
    @Autowired
    UserFeign userFeign;
    @Resource
    VehicleFeign vehicleFeign;
    @Autowired
    RabbitProduct rabbitProduct;

    @Autowired
    OrderMemberDetailBiz orderMemberDetailBiz;

    @Value("${gateway.insideHost}")
    String insideHost;

    @Value("${gateway.host}")
    String host;

    @Value("${gateway.order}")
    String order;


    String qrcodePrefix = "fcwx://order";
    String notifyUrl = "/baseOrder/app/unauth/notifyUrl";

    @RequestMapping(value = "/app/unauth/getOrderUnitPrice", method = RequestMethod.GET)
    @ResponseBody
    @ApiOperation(value = "获取订单相关的单价")
    @IgnoreClientToken
    @IgnoreUserToken
    public ObjectRestResponse<UnitPriceDTO> getOrderUnitPrice() {
        Map<String, Dictionary> dictionaryMap = thirdFeign.dictionaryGetAll4Map().getData();
        return ObjectRestResponse.succ(new UnitPriceDTO(){{
            setInsurePrice(new BigDecimal(dictionaryMap.get(APP_ORDER+ "_"+ DictionaryKey.INSURE_PRICE).getDetail()));
        }});
    }

    @RequestMapping(value = "/page", method = RequestMethod.GET)
    @ResponseBody
    @ApiOperation(value = "订单列表")
    @IgnoreClientToken
    public ObjectRestResponse<PageDataVO<OrderPageVO>> list(QueryOrderDTO dto) {
        //查询列表数据
        if (StringUtils.isBlank(BaseContextHandler.getUserID())) {
            throw new BaseException(ResultCode.AJAX_WECHAT_NOTEXIST_CODE);
        }
        dto.setCrtUser(Integer.valueOf(BaseContextHandler.getUserID()));
        Query query = new Query(dto);
        PageDataVO<OrderPageVO> pages = PageDataVO.pageInfo(query, () -> baseOrderBiz.pageByParm(query.getSuper()));
        pages.getData().parallelStream().forEach(data -> data.setQrcodeStr(qrcodePrefix));
        return new ObjectRestResponse<>().data(pages);
    }

    @RequestMapping(value = "/getOrderList", method = RequestMethod.GET)
    @ResponseBody
    @ApiOperation(value = "获取后台订单列表")
    @IgnoreClientToken
    public ObjectRestResponse<PageDataVO<OrderPageVO>> selectByUser(QueryOrderList dto) {
        if (dto.getStartTime() != null) {
            if (dto.getEndTime() == null) {
                dto.setEndTime(new Date().getTime());
            }
        }
        Query query = new Query(dto);
        PageDataVO pageDataVO = PageDataVO.pageInfo(query, () -> baseOrderBiz.listOrder(query.getSuper()));
        List<OrderPageVO> list = pageDataVO.getData();
        List<Integer> userIds = list.stream().map(OrderPageVO::getUserId).distinct().collect(Collectors.toList());
        ObjectRestResponse<List<AppUserVo>> objectRestResponse = userFeign.getByUserIds(userIds);
        for (OrderPageVO orderPageVO : list) {
            if (objectRestResponse.getData() != null && objectRestResponse.getData().size() > 0) {
                for (AppUserVo appUserVo : objectRestResponse.getData()) {
                    if (appUserVo != null && appUserVo.getUserid() == orderPageVO.getUserId()) {
                        orderPageVO.setUsername(appUserVo.getNickname());
                    }
                }
            }
            if (orderPageVO.getOrderRentVehicleDetail() != null && orderPageVO.getOrderRentVehicleDetail().getVehicleId() != null) {
                RestResponse<Vehicle> restResponse = vehicleFeign.findById(orderPageVO.getOrderRentVehicleDetail().getVehicleId());
                log.info("获取车辆信息返回消息：{}", restResponse.getMessage());
                if (restResponse.getData() != null) {
                    orderPageVO.setVehicalNumberPlat(restResponse.getData().getNumberPlate());
                }
            }
        }

        return new ObjectRestResponse<>().data(pageDataVO);
    }

    @RequestMapping(value = "/getVehicleList", method = RequestMethod.GET)
    @ResponseBody
    @ApiOperation(value = "交还车租车订单列表")
    @IgnoreClientToken
    public ObjectRestResponse<List<OrderPageVO>> getVehicleList(QueryOrderList dto) {
        if (dto.getStartTime() != null) {
            if (dto.getEndTime() == null) {
                dto.setEndTime(new Date().getTime());
            }
        }
        Query query = new Query(dto);
        PageDataVO pageDataVO = PageDataVO.pageInfo(query, () -> baseOrderBiz.getRentVehicle(query.getSuper()));
        return new ObjectRestResponse<>().data(pageDataVO);
    }

    @RequestMapping(value = "/getTourList", method = RequestMethod.GET)
    @ResponseBody
    @ApiOperation(value = "交还车旅游订单列表")
    @IgnoreClientToken
    public ObjectRestResponse<List<OrderPageVO>> getTourList(QueryOrderList dto) {
        if (dto.getStartTime() != null) {
            if (dto.getEndTime() == null) {
                dto.setEndTime(new Date().getTime());
            }

        }
        Query query = new Query(dto);
        PageDataVO pageDataVO = PageDataVO.pageInfo(query, () -> baseOrderBiz.getTourList(query.getSuper()));
        return new ObjectRestResponse<>().data(pageDataVO);
    }

    @ApiOperation("订单详情")
    @RequestMapping(value = "/{no}", method = RequestMethod.GET)
    @IgnoreClientToken
    @ResponseBody
    public ObjectRestResponse<OrderPageVO> get(@PathVariable String no) {
        //查询列表数据
        if (StringUtils.isBlank(BaseContextHandler.getUserID())) {
            throw new BaseException(ResultCode.AJAX_WECHAT_NOTEXIST_CODE);
        }

        Query query = initQuery(no);
        PageDataVO<OrderPageVO> page = PageDataVO.pageInfo(query, () -> baseOrderBiz.pageByParm(query.getSuper()));
        if (page.getData().isEmpty()) {
            throw new BaseException(ResultCode.NOTEXIST_CODE);
        }
        return new ObjectRestResponse<>().data(page.getData().get(0));
    }

    @GetMapping(value = "/orderDetail")
    @ApiOperation(value = "交还车获取订单详情，带驾驶人信息")
    @IgnoreClientToken
    @ResponseBody
    public ObjectRestResponse getOrderDetail(String orderNo) {
        if(StringUtils.isBlank(orderNo)) {
            return ObjectRestResponse.paramIsEmpty();
        }
        return baseOrderBiz.getOrderDetail(orderNo);
    }


    @ApiOperation("结合后台人员所属公司查询订单详情")
    @RequestMapping(value = "/company/{no}", method = RequestMethod.GET)
    @IgnoreClientToken
    @ResponseBody
    public ObjectRestResponse<OrderPageVO> companyGet(@PathVariable String no) {
        //查询列表数据
        if (StringUtils.isBlank(BaseContextHandler.getUserID())) {
            throw new BaseException(ResultCode.AJAX_WECHAT_NOTEXIST_CODE);
        }

        Query query = initCompanyQuery(no);
        PageDataVO<OrderPageVO> page = PageDataVO.pageInfo(query, () -> baseOrderBiz.pageByParm(query.getSuper()));
        if (page.getData().isEmpty()) {
            throw new BaseException(ResultCode.NOTEXIST_CODE);
        }
        return new ObjectRestResponse<>().data(page.getData().get(0));
    }

    private Query initQuery(String no) {
        QueryOrderDetailDTO qodd = new QueryOrderDetailDTO();
        qodd.setCrtUser(Integer.valueOf(BaseContextHandler.getUserID()));
        qodd.setNo(no);
        qodd.setLimit(1);
        qodd.setPage(1);
        Query query = new Query(qodd);
        return query;
    }

    private Query initCompanyQuery(String no) {
        QueryOrderDetailDTO qodd = new QueryOrderDetailDTO();
        //查询公司id
        UserDTO userDto = userFeign.userinfoByToken(BaseContextHandler.getToken()).getData();
        qodd.setCrtCompanyId(userDto.getCompanyId());
        qodd.setNo(no);
        qodd.setLimit(1);
        qodd.setPage(1);
        Query query = new Query(qodd);
        return query;
    }

    @RequestMapping(value = "/pay", method = RequestMethod.POST)
    @ResponseBody
    @ApiOperation(value = "订单支付")
    @IgnoreClientToken
    public ObjectRestResponse pay(@RequestBody RentVehicleOrderPayVO vo) {
        //查询列表数据
        if (StringUtils.isBlank(BaseContextHandler.getUserID())) {
            throw new BaseException(ResultCode.AJAX_WECHAT_NOTEXIST_CODE);
        }

        OrderPayVo orderPayVo = BeanUtil.toBean(vo, OrderPayVo.class);

        BaseOrder baseOrder = baseOrderBiz.selectOne(new BaseOrder() {{
            setNo(vo.getOrderNo());
        }});

        if (null == baseOrder || !OrderStatusEnum.ORDER_UNPAY.getCode().equals(baseOrder.getStatus())) {
            throw new BaseException("订单不可支付", ResultCode.EXCEPTION_CODE);
        }

        orderPayVo.setAmount(baseOrder.getRealAmount().multiply(new BigDecimal("100")).intValue());
        orderPayVo.setBody(baseOrder.getName());
        orderPayVo.setBuyerIp(StringUtils.defaultIfBlank(request.getHeader("userHost"), ClientUtil.getClientIp(request)));
        orderPayVo.setChannel(baseOrder.getType());
        orderPayVo.setNotifyUrl(insideHost + order + notifyUrl + "?orderNo=" + baseOrder.getNo());
        orderPayVo.setSubject(baseOrder.getName());
        orderPayVo.setUserId(baseOrder.getUserId());
        ObjectRestResponse response = BeanUtil.toBean(thirdFeign.wx(orderPayVo), ObjectRestResponse.class);
        return response;
    }

    @RequestMapping(value = "/cancel/{no}", method = RequestMethod.POST)
    @ResponseBody
    @ApiOperation(value = "取消订单")
    @IgnoreClientToken
    public ObjectRestResponse cancel(@PathVariable String no, @RequestBody CancelOrderDTO cancelOrderDto) {
        //查询列表数据
        if (StringUtils.isBlank(BaseContextHandler.getUserID())) {
            throw new BaseException(ResultCode.AJAX_WECHAT_NOTEXIST_CODE);
        }
        BaseOrder dbBaseOrder = baseOrderBiz.selectOne(new BaseOrder() {{
            setNo(no);
        }});
        if (null == dbBaseOrder || !BaseContextHandler.getUserID().equals(dbBaseOrder.getUserId().toString())) {
            throw new BaseException(ResultCode.NOTEXIST_CODE);

        }
        dbBaseOrder.setCancelReason(cancelOrderDto.getCancelReason());
        baseOrderBiz.cancel(dbBaseOrder);
        return ObjectRestResponse.succ();
    }

    @RequestMapping(value = "/app/unauth/notifyUrl", method = RequestMethod.GET)
    @ResponseBody
    @ApiOperation(value = "订单支付回调")
    @IgnoreClientToken
    @IgnoreUserToken
    public ObjectRestResponse notifyUrl(NotifyUrlDTO dto) {
        //判断是否成功，并且订单是未支付状态，则添加支付编号和修改状态
        if (StrUtil.isNotBlank(dto.getOrderNo())) {
            baseOrderBiz.payNotifyHandle(dto.getOrderNo(), dto.getTradeNo());
        }else{
            log.error(" exception notifyUrl : " + JSONUtil.toJsonStr(dto));
        }
        return ObjectRestResponse.succ();
    }

    @RequestMapping(value = "/app/unauth/send-msg", method = RequestMethod.GET)
    @ResponseBody
    @ApiOperation(value = "mq发送消息")
    @IgnoreClientToken
    @IgnoreUserToken
    public ObjectRestResponse sendMsg() {
        rabbitProduct.sendDelayMessage(baseOrderBiz.selectById(93), 1000L * 60);
        return ObjectRestResponse.succ();
    }

    /**
     * 订单查询类
     */
    @Data
    public class QueryOrderDTO extends PageParam {
        @ApiModelProperty(hidden = false)
        private Integer crtUser;

        @ApiModelProperty(hidden = false)
        private Integer crtCompanyId;

        @ApiModelProperty(value = "订单状态\n" +
                "0--删除\n" +
                "1--创建订单\n" +
                "2--取消\n" +
                "3--待付款\n" +
                "4--待出行\n" +
                "5--出行中（进行中）\n" +
                "6--已完成")
        private Integer status;

        private String multiStatus;

        @ApiModelProperty(value = "是否有使用会员权益")
        private Integer hasMemberRight;

        @ApiModelProperty(value = "订单类型 1--租车；2--旅游; 3--会员订单")
        private Integer type;
    }

    /**
     * 订单查询类
     */
    @Data
    public class QueryOrderDetailDTO extends QueryOrderDTO {
        private String no;
    }

    @Data
    public class QueryOrderList extends QueryOrderDetailDTO {

        private Long startTime;

        private Long endTime;

        private Integer startCompanyId;

        private Integer startZoneId;

        private Integer userId;

        private Integer type;
    }

    @Data
    public class NotifyUrlDTO {
        //订单号
        @ApiModelProperty(value = "订单号")
        private String orderNo;

        //订单号
        @ApiModelProperty(value = "支付订单（流水）号")
        private String tradeNo;
    }
    }
