package com.xxfc.platform.order.biz;

import cn.hutool.json.JSONUtil;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xxfc.platform.order.contant.enumerate.AccountTypeEnum;
import com.xxfc.platform.order.contant.enumerate.DeductionTypeEnum;
import com.xxfc.platform.order.contant.enumerate.OrderTypeEnum;
import com.xxfc.platform.order.entity.DailyOrderStatistics;
import com.xxfc.platform.order.entity.DailyVehicleOrderStatistics;
import com.xxfc.platform.order.entity.OrderAccount;
import com.xxfc.platform.order.entity.OrderStatistics;
import com.xxfc.platform.order.mapper.DailyVehicleOrderStatisticsMapper;
import com.xxfc.platform.order.pojo.Term;
import com.xxfc.platform.order.pojo.account.OrderAccountDTO;
import com.xxfc.platform.order.pojo.account.OrderAccountDeduction;
import com.xxfc.platform.order.pojo.account.OrderAccountDetail;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.collections4.Put;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import tk.mybatis.mapper.common.example.UpdateByExampleSelectiveMapper;
import tk.mybatis.mapper.entity.Example;
import tk.mybatis.mapper.weekend.WeekendSqls;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 每日租车订单统计
 *
 * @author Administrator
 */
@Service
@Slf4j
public class DailyVehicleOrderStatisticsBiz extends BaseBiz<DailyVehicleOrderStatisticsMapper, DailyVehicleOrderStatistics> {
    /**
     * 延期
     */
    private final Integer TYPE_DEFERRED = 1;
    /**
     * 损坏
     */
    private final Integer TYPE_DAMAGE = 2;
    /**
     * 其他
     */
    private final Integer TYPE_OTHER = 3;

    @Autowired
    private OrderAccountBiz accountBiz;


    /**
     * 根据type获取对应的金额总和
     *
     * @param list
     * @param types 金额类型
     * @return
     */
    private BigDecimal get(List<OrderAccountDeduction> list, List<Integer> types) {
        return list.parallelStream()
                .filter(el -> types.contains(el.getType()))
                .map(OrderAccountDeduction::getAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

    }

    /***
     * 添加每日记录
     * @param entity
     * @return
     */
    @Override
    public int insertSelectiveRe(DailyVehicleOrderStatistics entity) {
        entity.setCrtTime(new Date());
        return mapper.insertSelective(entity);
    }


    public OrderStatistics findAll(List<Integer> companyIds) {
        return mapper.monthOrderTotal(companyIds);
    }

    /**
     * 统计订单并保存到数据库当中
     *
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean dailyVehicleOrderRecord(Integer day) {
        try {
            //获取每日订单统计
            List<DailyVehicleOrderStatistics> orderStatistics = getDailyVehicleOrderRecord(new Term(OrderTypeEnum.RENT_VEHICLE.getCode(),day,1,null,null,null));
            log.info("Vehicle:统计完成");
            save(orderStatistics);
            log.info("Vehicle:保存成功");
            return true;
        } catch (Exception e) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            log.error("Vehicle:" + e.getMessage());
            return false;
        }
    }

    public void save(List<DailyVehicleOrderStatistics> orderStatistics) {
        if (CollectionUtils.isNotEmpty(orderStatistics)) {
            for (DailyVehicleOrderStatistics orderStatistic : orderStatistics) {
                //保存统计
                if (JudgmentOfExistence(orderStatistic)) {

                    insertSelectiveRe(orderStatistic);
                } else {
                    mapper.updateByExampleSelective(orderStatistic, Example.builder(DailyVehicleOrderStatistics.class)
                            .where(WeekendSqls.<DailyVehicleOrderStatistics>custom()
                                    .andEqualTo(DailyVehicleOrderStatistics::getOneDay, orderStatistic.getOneDay())
                                    .andEqualTo(DailyVehicleOrderStatistics::getBranchCompanyId, orderStatistic.getBranchCompanyId()))
                            .build());
                }
            }
        }
    }

    private boolean JudgmentOfExistence(DailyVehicleOrderStatistics orderStatistic) {
        List<DailyVehicleOrderStatistics> statistics = selectByExample(Example.builder(DailyVehicleOrderStatistics.class)
                .where(WeekendSqls.<DailyVehicleOrderStatistics>custom()
                        .andEqualTo(DailyVehicleOrderStatistics::getOneDay, orderStatistic.getOneDay())
                        .andEqualTo(DailyVehicleOrderStatistics::getBranchCompanyId, orderStatistic.getBranchCompanyId()))
                .build());
        if (CollectionUtils.isNotEmpty(statistics)) {
            return false;
        }
        return true;
    }

    /**
     * 获取所有分公司每日统计记录
     *
     * @return
     * @param term 查询条件
     */
    public List<DailyVehicleOrderStatistics> getDailyVehicleOrderRecord(Term term) {
        //获取当天所有订单账目
        List<OrderAccountDTO> accountList = accountBiz.getOrderAccountByOrderType(term);
        if (CollectionUtils.isEmpty(accountList)) {
            return null;
        }
        //获取每个分公司当天所有的订单帐目
        HashMap<Integer, List<OrderAccountDTO>> companyIdmap = new HashMap<>();
        for (OrderAccountDTO orderAccountDTO : accountList) {
            List<OrderAccountDTO> list = companyIdmap.get(orderAccountDTO.getCompanyId());
            if (list == null) {
                list = new ArrayList<>();
                companyIdmap.put(orderAccountDTO.getCompanyId(), list);
            }
            list.add(orderAccountDTO);
        }

        //获取每个公司当天订单的总数据
        ArrayList<DailyVehicleOrderStatistics> statistics = new ArrayList<>();
        for (Map.Entry<Integer, List<OrderAccountDTO>> integerListEntry : companyIdmap.entrySet()) {
            //获取不同类型的账单详情
            List<OrderAccountDTO> orderAccountDTOS = integerListEntry.getValue();
            if (CollectionUtils.isNotEmpty(orderAccountDTOS)) {
                Map<Integer, List<OrderAccountDTO>> typeMap = Maps.newHashMap();
                for (OrderAccountDTO orderAccountDTO : orderAccountDTOS) {

                    List<OrderAccountDTO> list = typeMap.get(orderAccountDTO.getAccountType());
                    if (list == null) {
                        list = new ArrayList<>();
                        typeMap.put(orderAccountDTO.getAccountType(), list);

                    }
                    list.add(orderAccountDTO);
                }

                if (MapUtils.isNotEmpty(typeMap)) {
                    //获取该公司当天的账单数据
                    if (CollectionUtils.isNotEmpty(orderAccountDTOS)) {
                        DailyVehicleOrderStatistics dvos = getStatisticsByOrderAccountDetail(typeMap);
                        if (dvos != null) {
                            statistics.add(dvos);
                            dvos.setBranchCompanyId(integerListEntry.getKey());
                            dvos.setOneDay(orderAccountDTOS.get(0).getOneDay());
                        }
                    }
                }
            }
        }

        return statistics;
    }


    /**
     * 获取该分公司的统计数据
     *
     * @param map
     * @return
     */
    private DailyVehicleOrderStatistics getStatisticsByOrderAccountDetail(Map<Integer, List<OrderAccountDTO>> map) {
        //获取收入金额
        DailyVehicleOrderStatistics orderStatistics = getGmvAndSecurityDeposit(map);
        //设置补偿总额
        orderStatistics.setOrderCompensation(getOrderCompensation(map));
        //获取扣款和退还金额
        refundAndDeductions(map, orderStatistics);
        return orderStatistics;
    }

    /**
     *
     * 获取订单补偿金额(该状态发生在退还部分款时,补偿订单款=单实际产生订单款+退还-支付订单款)
     * @param map
     * @return
     */
    private BigDecimal getOrderCompensation(Map<Integer, List<OrderAccountDTO>> map) {
        //获取当天部分押金退款(203)数据
        List<OrderAccountDTO> orderAccountDTOS = map.get((AccountTypeEnum.OUT_PART_DEPOSIT.getCode()));
        if (CollectionUtils.isEmpty(orderAccountDTOS)) {
            return BigDecimal.ZERO;
        }
        ArrayList<OrderAccountDetail> orderAccountDetail = getOrderAccountDetail(orderAccountDTOS);
        //获取订单支付总额
        BigDecimal payment = orderAccountDetail
                .parallelStream()
                .map(OrderAccountDetail::getOriginOrderAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        //获取订单退还总额
        BigDecimal giveBack = orderAccountDetail
                .parallelStream()
                .map(OrderAccountDetail::getOrderAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        //获取明细集合
        ArrayList<OrderAccountDeduction> orderAccountDeductions = Lists.newArrayList();
                 orderAccountDetail
                .parallelStream()
                .map(OrderAccountDetail::getDeductions)
                .forEach(list->orderAccountDeductions.addAll(list));
        //获取订单实际款
        BigDecimal actual = orderAccountDeductions
                .parallelStream()
                .filter(o -> DeductionTypeEnum.CONSUME.getCode().equals(o.getType()))
                .map(OrderAccountDeduction::getAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
        return actual.add(giveBack).subtract(payment);


    }

    /**
     * 获取扣款和退款金额
     *
     * @param map
     * @param orderStatistics
     */
    private void refundAndDeductions(Map<Integer, List<OrderAccountDTO>> map, DailyVehicleOrderStatistics orderStatistics) {
        ArrayList<OrderAccountDTO> arrayList = Lists.newArrayList();
        for (Integer key : map.keySet()) {
            if (!key.equals(AccountTypeEnum.IN_ORDER_PAY.getCode())) {
                arrayList.addAll(map.get(key));
            }
        }
        if (CollectionUtils.isNotEmpty(arrayList)) {
            ArrayList<OrderAccountDetail> orderAccountDetail = getOrderAccountDetail(arrayList);
            if (CollectionUtils.isNotEmpty(orderAccountDetail)) {
                //获取退还订总额
                BigDecimal returnGmv = getOrderAmountTotal(orderAccountDetail);
                orderStatistics.setReturnGmv(returnGmv);
                //获取退还押金总额
                BigDecimal refundSecurityDeposit = getDepositAmountTotal(orderAccountDetail);
                orderStatistics.setRefundSecurityDeposit(refundSecurityDeposit);
                List<OrderAccountDeduction> OrderAccountDeductions = gettDeductions(orderAccountDetail);
                //获取违章总额
                orderStatistics.setForfeit(get(OrderAccountDeductions, new ArrayList<Integer>() {{
                    add(DeductionTypeEnum.VIOLATE_TRAFFIC_DEDUCT.getCode());
                }}));
                //获取损坏赔偿总额
                orderStatistics.setCompensation(get(OrderAccountDeductions, new ArrayList<Integer>() {{
                    add(DeductionTypeEnum.DAMAGES.getCode());
                }}));
                //获取违约总额
                ArrayList<Integer> types = Lists.newArrayList();
                types.add(DeductionTypeEnum.VIOLATE_ADVANCE.getCode());
                types.add(DeductionTypeEnum.VIOLATE_CANCEL.getCode());
                types.add(DeductionTypeEnum.VIOLATE_DELAY.getCode());
                orderStatistics.setDefaultMoney(get(OrderAccountDeductions, types));


                //消费金额
//                orderStatistics.setTotalExpenditure(get(OrderAccountDeductions, new ArrayList<Integer>() {{
//                    add(DeductionTypeEnum.CONSUME.getCode());
//                }}));
            }
        }
    }

    /**
     * 获取订单总额和预交押金总额
     *
     * @param map
     * @return
     */
    private DailyVehicleOrderStatistics getGmvAndSecurityDeposit(Map<Integer, List<OrderAccountDTO>> map) {
        DailyVehicleOrderStatistics orderStatistics = new DailyVehicleOrderStatistics();
        List<OrderAccountDTO> orderAccountDTOS = map.get(AccountTypeEnum.IN_ORDER_PAY.getCode());

        if (CollectionUtils.isNotEmpty(orderAccountDTOS)) {
            ArrayList<OrderAccountDetail> orderAccountDetails = getOrderAccountDetail(orderAccountDTOS);
            if (CollectionUtils.isNotEmpty(orderAccountDetails)) {
                //获取订单总额
                BigDecimal gmv = getOrderAmountTotal(orderAccountDetails);
                orderStatistics.setGmv(gmv);
                //获取押金总额
                BigDecimal securityDeposit = getDepositAmountTotal(orderAccountDetails);
                orderStatistics.setSecurityDeposit(securityDeposit);


            }
        }
        return orderStatistics;
    }

    /**
     * 获取订单和
     *
     * @param orderAccountDetails
     * @return
     */
    private BigDecimal getDepositAmountTotal(ArrayList<OrderAccountDetail> orderAccountDetails) {
        return orderAccountDetails.stream()
                .map(OrderAccountDetail::getDepositAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 获取押金和
     *
     * @param orderAccountDetails
     * @return
     */
    private BigDecimal getOrderAmountTotal(ArrayList<OrderAccountDetail> orderAccountDetails) {
        return orderAccountDetails.stream()
                .map(OrderAccountDetail::getOrderAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 获取赔款数据
     *
     * @param orderAccountDetails
     * @return
     */
    private List<OrderAccountDeduction> gettDeductions(ArrayList<OrderAccountDetail> orderAccountDetails) {
        ArrayList<OrderAccountDeduction> arrayList = Lists.newArrayList();
        orderAccountDetails.parallelStream().map(OrderAccountDetail::getDeductions).forEach(e -> arrayList.addAll(e));
        return arrayList;
    }

    /**
     * 获取金额详情
     *
     * @param orderAccountDTOS
     * @return
     */
    private ArrayList<OrderAccountDetail> getOrderAccountDetail(List<OrderAccountDTO> orderAccountDTOS) {
        ArrayList<OrderAccountDetail> arrayList = Lists.newArrayList();
        orderAccountDTOS.parallelStream()
                .map(OrderAccount::getAccountDetail)
                .forEach(detail -> {
                    if (StringUtils.isNotBlank(detail)) {
                        arrayList.add(JSONUtil.toBean(detail, OrderAccountDetail.class));
                    }
                });

        return arrayList;
    }


}
