package com.xxfc.platform.order.biz;

import cn.hutool.core.bean.BeanUtil;
import cn.hutool.core.bean.copier.CopyOptions;
import cn.hutool.json.JSONUtil;
import com.github.wxiaoqi.security.common.biz.BaseBiz;
import com.github.wxiaoqi.security.common.exception.BaseException;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.xxfc.platform.order.contant.enumerate.AccountTypeEnum;
import com.xxfc.platform.order.contant.enumerate.DeductionTypeEnum;
import com.xxfc.platform.order.contant.enumerate.OrderTypeEnum;
import com.xxfc.platform.order.entity.*;
import com.xxfc.platform.order.mapper.DailyTravelOrderStatisticsMapper;
import com.xxfc.platform.order.pojo.Term;
import com.xxfc.platform.order.pojo.account.OrderAccountDTO;
import com.xxfc.platform.order.pojo.account.OrderAccountDeduction;
import com.xxfc.platform.order.pojo.account.OrderAccountDetail;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.transaction.interceptor.TransactionAspectSupport;
import tk.mybatis.mapper.entity.Example;
import tk.mybatis.mapper.weekend.WeekendSqls;

import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 每日旅游订单统计
 *
 * @author Administrator
 */
@Service
@Slf4j
public class DailyTravelOrderStatisticsBiz extends BaseBiz<DailyTravelOrderStatisticsMapper, DailyTravelOrderStatistics> {


    @Autowired
    private OrderAccountBiz accountBiz;

    /**
     * 统计订单并保存到数据库当中
     * @return
     */
    @Transactional(rollbackFor = Exception.class)
    public boolean dailyTravelOrderRecord(Integer day) {
        try {
            //获取每日订单统计
            List<DailyTravelOrderStatistics> orderStatistics = getDailyTravelOrderStatistics(new Term(OrderTypeEnum.TOUR.getCode(),day,1,null,null,null));
            log.info("Travel:统计完成");
            seve(orderStatistics);
            log.info("Travel:保存成功");
            return true;
        } catch (Exception e) {
            TransactionAspectSupport.currentTransactionStatus().setRollbackOnly();
            log.error("Travel:"+e.getMessage());
            return false;
        }
    }

    public void seve(List<DailyTravelOrderStatistics> orderStatistics) {
        if (CollectionUtils.isNotEmpty(orderStatistics)) {
            for (DailyTravelOrderStatistics orderStatistic : orderStatistics) {
                if (JudgmentOfExistence(orderStatistic)) {

                    insertSelectiveRe(orderStatistic);
                } else {
                    mapper.updateByExampleSelective(orderStatistic, Example.builder(DailyTravelOrderStatistics.class)
                            .where(WeekendSqls.<DailyTravelOrderStatistics>custom()
                                    .andEqualTo(DailyTravelOrderStatistics::getOneDay, orderStatistic.getOneDay())
                                    .andEqualTo(DailyTravelOrderStatistics::getBranchCompanyId, orderStatistic.getBranchCompanyId()))
                            .build());
                }
            }
        }
    }

    /**
     * 添加前查询是否存在这条数据
     * @param orderStatistic
     * @return
     */
    private boolean JudgmentOfExistence(DailyTravelOrderStatistics orderStatistic) {
        List<DailyTravelOrderStatistics> statistics = selectByExample(Example.builder(DailyTravelOrderStatistics.class)
                .where(WeekendSqls.<DailyTravelOrderStatistics>custom()
                        .andEqualTo(DailyTravelOrderStatistics::getOneDay, orderStatistic.getOneDay())
                        .andEqualTo(DailyTravelOrderStatistics::getBranchCompanyId, orderStatistic.getBranchCompanyId()))
                .build());
        if (CollectionUtils.isNotEmpty(statistics)) {
            return false;
        }
        return true;
    }
    public List<DailyTravelOrderStatistics> getDailyTravelOrderStatistics(Term term) {
        //获取当天所有订单账目
        List<OrderAccountDTO> accountList = accountBiz.getOrderAccountByOrderType(term);
        if (CollectionUtils.isEmpty(accountList)) {
            return null;
        }
        //获取每个分公司,当天所有的订单帐目
        HashMap<Integer, List<OrderAccountDTO>> companyIdmap = new HashMap<>();
        for (OrderAccountDTO orderAccountDTO : accountList) {
            List<OrderAccountDTO> list = companyIdmap.get(orderAccountDTO.getCompanyId());
            if (list == null) {
                list = new ArrayList<>();
                companyIdmap.put(orderAccountDTO.getCompanyId(), list);
            }
            list.add(orderAccountDTO);
        }

        //获取每个公司当天订单的总数据
        ArrayList<DailyTravelOrderStatistics> statistics = new ArrayList<>();
        for (Map.Entry<Integer, List<OrderAccountDTO>> integerListEntry : companyIdmap.entrySet()) {
            //获取不同类型的账单详情
            List<OrderAccountDTO> orderAccountDTOS = integerListEntry.getValue();
            if (CollectionUtils.isNotEmpty(orderAccountDTOS)) {
                Map<Integer, List<OrderAccountDTO>> typeMap = Maps.newHashMap();
                for (OrderAccountDTO orderAccountDTO : orderAccountDTOS) {
                    List<OrderAccountDTO> list = typeMap.get(orderAccountDTO.getAccountType());
                    if (list == null) {
                        list = new ArrayList<>();
                        typeMap.put(orderAccountDTO.getAccountType(), list);
                    }
                    list.add(orderAccountDTO);
                }

                if (MapUtils.isNotEmpty(typeMap)) {
                    //获取该公司当天的账单数据
                    if (CollectionUtils.isNotEmpty(orderAccountDTOS)) {
                        DailyTravelOrderStatistics dvos = getStatisticsByOrderAccountDetail(typeMap);
                        if (dvos != null) {
                            statistics.add(dvos);
                            dvos.setBranchCompanyId(integerListEntry.getKey());
                            dvos.setOneDay(orderAccountDTOS.get(0).getOneDay());
                        }
                    }
                }
            }
        }

        return statistics;
    }

    private DailyTravelOrderStatistics getStatisticsByOrderAccountDetail(Map<Integer, List<OrderAccountDTO>> map) {
        //获取收入金额
        DailyTravelOrderStatistics orderStatistics = getGmvAndSecurityDeposit(map);
        //获取扣款和退还金额
        refundAndDeductions(map, orderStatistics);
        return orderStatistics;
        
    }

    /**
     * 获取扣款和退款金额
     *
     * @param map
     * @param orderStatistics
     */
    private void refundAndDeductions(Map<Integer, List<OrderAccountDTO>> map, DailyTravelOrderStatistics orderStatistics) {
        ArrayList<OrderAccountDTO> arrayList = Lists.newArrayList();
        for (Integer key : map.keySet()) {
            if (!key.equals(AccountTypeEnum.IN_ORDER_PAY.getCode())) {
                arrayList.addAll(map.get(key));
            }
        }
        if (CollectionUtils.isNotEmpty(arrayList)) {
            ArrayList<OrderAccountDetail> orderAccountDetail = getOrderAccountDetail(arrayList);


            if (CollectionUtils.isNotEmpty(orderAccountDetail)) {
                //订单退还总额
                BigDecimal depositAmountTotal = getOrderAmountTotal(orderAccountDetail);
                orderStatistics.setReturnGmv(depositAmountTotal);
                List<OrderAccountDeduction> OrderAccountDeductions = gettDeductions(orderAccountDetail);
                //获取违约总额
                BigDecimal bigDecimal = get(OrderAccountDeductions, new ArrayList<Integer>() {{
                    add(DeductionTypeEnum.VIOLATE_CANCEL.getCode());
                    add(DeductionTypeEnum.VIOLATE_ADVANCE.getCode());
                    add(DeductionTypeEnum.VIOLATE_DELAY.getCode());
                }});
                orderStatistics.setDefaultMoney(bigDecimal);

            }
        }
    }




    
    private DailyTravelOrderStatistics getGmvAndSecurityDeposit(Map<Integer, List<OrderAccountDTO>> map) {
        DailyTravelOrderStatistics orderStatistics = new DailyTravelOrderStatistics();
        List<OrderAccountDTO> orderAccountDTOS = map.get(AccountTypeEnum.IN_ORDER_PAY.getCode());

        if (CollectionUtils.isNotEmpty(orderAccountDTOS)) {
            ArrayList<OrderAccountDetail> orderAccountDetails = getOrderAccountDetail(orderAccountDTOS);
            if (CollectionUtils.isNotEmpty(orderAccountDetails)) {
                //获取订单总额
                BigDecimal gmv = getOrderAmountTotal(orderAccountDetails);
                orderStatistics.setGmv(gmv);

            }
        }
        return orderStatistics;
    }


    /**
     * 获取金额详情
     *
     * @param orderAccountDTOS
     * @return
     */
    private ArrayList<OrderAccountDetail> getOrderAccountDetail(List<OrderAccountDTO> orderAccountDTOS) {
        ArrayList<OrderAccountDetail> arrayList = Lists.newArrayList();
        orderAccountDTOS.parallelStream()
                .map(OrderAccount::getAccountDetail)
                .forEach(detail -> {
                    if (StringUtils.isNotBlank(detail)) {
                        arrayList.add(JSONUtil.toBean(detail, OrderAccountDetail.class));
                    }
                });

        return arrayList;
    }


    /**
     * 获取订单和
     *
     * @param orderAccountDetails
     * @return
     */
    private BigDecimal getOrderAmountTotal(ArrayList<OrderAccountDetail> orderAccountDetails) {
        return orderAccountDetails.stream()
                .map(OrderAccountDetail::getOrderAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    /**
     * 获取赔款数据
     *
     * @param orderAccountDetails
     * @return
     */
    private List<OrderAccountDeduction> gettDeductions(ArrayList<OrderAccountDetail> orderAccountDetails) {
        ArrayList<OrderAccountDeduction> arrayList = Lists.newArrayList();
        orderAccountDetails.parallelStream().map(OrderAccountDetail::getDeductions).forEach(e->arrayList.addAll(e));
        return arrayList;
    }


    /**
     * 根据type获取对应的金额总和
     *
     * @param list
     * @param type                  金额类型
     * @return
     */
    private BigDecimal get(List<OrderAccountDeduction> list, List<Integer> type) {
        return list.parallelStream()
                .filter(el ->type.contains(el.getType()))
                .map(OrderAccountDeduction::getAmount)
                .reduce(BigDecimal.ZERO,BigDecimal::add);

    }

    @Override
    public int insertSelectiveRe(DailyTravelOrderStatistics entity) {
        entity.setCrtTime(new Date());
        return mapper.insertSelective(entity);
    }

    public OrderStatistics findAll(List<Integer> companyIds) {
        return mapper.monthOrderTotal(companyIds);
    }
}
