package com.base.utils.tools.android;

import android.content.Context;
import android.text.TextUtils;

import com.amap.api.location.AMapLocation;
import com.amap.api.location.AMapLocationClient;
import com.amap.api.location.AMapLocationClientOption;
import com.amap.api.location.AMapLocationListener;

/**
 * 位置工具
 */

public class LocationUtil {
    private OnLocationListener onLocationListener;

    public interface OnLocationListener {
        void onLocation(AMapLocation loc);
    }

    private Context context;

    public LocationUtil(Context context, OnLocationListener onLocationListener) {
        this.context = context;
        this.onLocationListener = onLocationListener;
        //初始化定位
        initLocation();
    }

    private AMapLocationClient locationClient = null;

    private AMapLocationClientOption locationOption = new AMapLocationClientOption();


    /**
     * 初始化定位
     */
    private void initLocation() {
        //初始化client
        locationClient = new AMapLocationClient(context.getApplicationContext());
        //设置定位参数
        locationClient.setLocationOption(getDefaultOption());
        // 设置定位监听
        locationClient.setLocationListener(locationListener);
    }


    /**
     * 默认的定位参数
     *
     * @author hongming.wang
     * @since 2.8.0
     */

    private AMapLocationClientOption getDefaultOption() {

        AMapLocationClientOption mOption = new AMapLocationClientOption();
        mOption.setLocationMode(AMapLocationClientOption.AMapLocationMode.Hight_Accuracy);                  //可选，设置定位模式，可选的模式有高精度、仅设备、仅网络。默认为高精度模式
        mOption.setGpsFirst(false);             //可选，设置是否gps优先，只在高精度模式下有效。默认关闭
        mOption.setHttpTimeOut(30000);          //可选，设置网络请求超时时间。默认为30秒。在仅设备模式下无效
        mOption.setInterval(2000);              //可选，设置定位间隔。默认为2秒
        mOption.setNeedAddress(true);           //可选，设置是否返回逆地理地址信息。默认是true
        mOption.setOnceLocation(false);         //可选，设置是否单次定位。默认是false
        mOption.setOnceLocationLatest(false);   //可选，设置是否等待wifi刷新，默认为false.如果设置为true,会自动变为单次定位，持续定位时不要使用
        AMapLocationClientOption.setLocationProtocol(AMapLocationClientOption.AMapLocationProtocol.HTTP);   //可选， 设置网络请求的协议。可选HTTP或者HTTPS。默认为HTTP
        mOption.setSensorEnable(false);         //可选，设置是否使用传感器。默认是false
        mOption.setWifiScan(true);              //可选，设置是否开启wifi扫描。默认为true，如果设置为false会同时停止主动刷新，停止以后完全依赖于系统刷新，定位位置可能存在误差
        mOption.setLocationCacheEnable(true);   //可选，设置是否使用缓存定位，默认为true
        return mOption;
    }


    /**
     * 定位监听
     */

    AMapLocationListener locationListener = new AMapLocationListener() {
        @Override
        public void onLocationChanged(AMapLocation loc) {
            if (onLocationListener != null) {
                onLocationListener.onLocation(loc);
            }
        }
    };


    // 根据控件的选择，重新设置定位参数

    private void resetOption() {

        // 设置是否需要显示地址信息
        locationOption.setNeedAddress(true);
        /**
         * 设置是否优先返回GPS定位结果，如果30秒内GPS没有返回定位结果则进行网络定位
         * 注意：只有在高精度模式下的单次定位有效，其他方式无效
         */

        locationOption.setGpsFirst(true);
        // 设置是否开启缓存
        locationOption.setLocationCacheEnable(true);
        // 设置是否单次定位
        locationOption.setOnceLocation(true);
        //设置是否等待设备wifi刷新，如果设置为true,会自动变为单次定位，持续定位时不要使用
        locationOption.setOnceLocationLatest(true);
        //设置是否使用传感器
        locationOption.setSensorEnable(true);
        //设置是否开启wifi扫描，如果设置为false时同时会停止主动刷新，停止以后完全依赖于系统刷新，定位位置可能存在误差
        String strInterval = "1000";
        if (!TextUtils.isEmpty(strInterval)) {
            try {
                // 设置发送定位请求的时间间隔,最小值为1000，如果小于1000，按照1000算
                locationOption.setInterval(Long.valueOf(strInterval));
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }

        String strTimeout = "10000";
        if (!TextUtils.isEmpty(strTimeout)) {
            try {
                // 设置网络请求超时时间
                locationOption.setHttpTimeOut(Long.valueOf(strTimeout));
            } catch (Throwable e) {
                e.printStackTrace();
            }
        }

    }


    /**
     * 开始定位
     *
     * @author hongming.wang
     * @since 2.8.0
     */

    public void startLocation() {
        // 设置定位参数
        locationClient.setLocationOption(locationOption);
        // 启动定位
        locationClient.startLocation();
    }


    /**
     * 停止定位
     *
     * @author hongming.wang
     * @since 2.8.0
     */

    public void stopLocation() {
        // 停止定位
        locationClient.stopLocation();
    }


    /**
     * 销毁定位
     *
     * @author hongming.wang
     * @since 2.8.0
     */

    public void destroyLocation() {

        if (null != locationClient) {
            /**
             * 如果AMapLocationClient是在当前Activity实例化的，
             * 在Activity的onDestroy中一定要执行AMapLocationClient的onDestroy
             */
            locationClient.onDestroy();
            locationClient = null;
            locationOption = null;
        }
    }
    /**
     响应码
     问题说明
     问题排查策略
     0
     定位成功。
     可以在定位回调里判断定位返回成功后再进行业务逻辑运算。
     1
     一些重要参数为空，如context；
     请对定位传递的参数进行非空判断。
     2
     定位失败，由于仅扫描到单个wifi，且没有基站信息。
     请重新尝试。
     3
     获取到的请求参数为空，可能获取过程中出现异常。
     请对所连接网络进行全面检查，请求可能被篡改。
     4
     请求服务器过程中的异常，多为网络情况差，链路不通导致
     请检查设备网络是否通畅，检查通过接口设置的网络访问超时时间，建议采用默认的30秒。
     5
     请求被恶意劫持，定位结果解析失败。
     您可以稍后再试，或检查网络链路是否存在异常。
     6
     定位服务返回定位失败。
     请获取errorDetail（通过getLocationDetail()方法获取）信息并参考定位常见问题进行解决。
     7
     KEY鉴权失败。
     请仔细检查key绑定的sha1值与apk签名sha1值是否对应，或通过高频问题查找相关解决办法。
     8
     Android exception常规错误
     请将errordetail（通过getLocationDetail()方法获取）信息通过工单系统反馈给我们。
     9
     定位初始化时出现异常。
     请重新启动定位。
     10
     定位客户端启动失败。
     请检查AndroidManifest.xml文件是否配置了APSService定位服务
     11
     定位时的基站信息错误。
     请检查是否安装SIM卡，设备很有可能连入了伪基站网络。
     12
     缺少定位权限。
     请在设备的设置中开启app的定位权限。
     13
     定位失败，由于未获得WIFI列表和基站信息，且GPS当前不可用。
     建议开启设备的WIFI模块，并将设备中插入一张可以正常工作的SIM卡，或者检查GPS是否开启；如果以上都内容都确认无误，请您检查App是否被授予定位权限。
     14
     GPS 定位失败，由于设备当前 GPS 状态差。
     建议持设备到相对开阔的露天场所再次尝试。
     15
     定位结果被模拟导致定位失败
     如果您希望位置被模拟，请通过setMockEnable(true);方法开启允许位置模拟
     16
     当前POI检索条件、行政区划检索条件下，无可用地理围栏
     建议调整检索条件后重新尝试，例如调整POI关键字，调整POI类型，调整周边搜区域，调整行政区关键字等。
     18
     定位失败，由于手机WIFI功能被关闭同时设置为飞行模式
     建议手机关闭飞行模式，并打开WIFI开关
     19
     定位失败，由于手机没插sim卡且WIFI功能被关闭
     建议手机插上sim卡，打开WIFI开关
     */
}