package com.rv.im.xmpp;

import android.media.MediaPlayer;
import android.os.Handler;
import android.os.Looper;
import android.util.Log;

import com.ruiwenliu.wrapper.util.AppUtils;
import com.rv.im.ImSetting;
import com.rv.im.R;
import com.rv.im.db.service.ChatMessageServiceImp;
import com.rv.im.db.service.FriendServiceImp;
import com.rv.im.db.service.IChatMessageService;
import com.rv.im.db.table.ChatMessage;
import com.rv.im.db.table.Friend;
import com.rv.im.enums.MessageTypeEnum;
import com.rv.im.presenter.AddFriendPresenter;
import com.rv.im.util.NotificationMananger;
import com.rv.im.xmpp.listener.AuthStateListener;
import com.rv.im.xmpp.listener.ChatMessageListener;
import com.rv.im.xmpp.listener.NewFriendListener;

import java.util.ArrayList;
import java.util.List;

public class ListenerManager {
    private static ListenerManager instance;
    /* 回调监听 */
    private List<AuthStateListener> mAuthStateListeners = new ArrayList<>();
    private List<ChatMessageListener> mChatMessageListeners = new ArrayList<>();
    private List<NewFriendListener> mNewFriendListeners = new ArrayList<>();

    private Handler mHandler = new Handler(Looper.getMainLooper());

    private ListenerManager() {
    }

    public static ListenerManager getInstance() {
        if (instance == null) {
            instance = new ListenerManager();
        }
        return instance;
    }

    public void reset() {
        instance = null;
    }

    /**********************
     * 注册和移除监听
     **************************/

    public void addAuthStateChangeListener(AuthStateListener authStateChangeListener) {
        mAuthStateListeners.add(authStateChangeListener);
    }

    public void removeAuthStateChangeListener(AuthStateListener authStateChangeListener) {
        mAuthStateListeners.remove(authStateChangeListener);
    }

    public void addChatMessageListener(ChatMessageListener messageListener) {
        mChatMessageListeners.add(messageListener);
    }

    public void removeChatMessageListener(ChatMessageListener messageListener) {
        mChatMessageListeners.remove(messageListener);
    }

    public void addNewFriendListener(NewFriendListener newFriendListener) {
        mNewFriendListeners.add(newFriendListener);
    }

    public void removeNewFriendListener(NewFriendListener friendListener) {
        mNewFriendListeners.remove(friendListener);
    }

    /**********************
     * 监听回调
     **************************/
    public void notifyAuthStateChange(final int authState) {
        if (mAuthStateListeners.size() <= 0) {
            return;
        }
        mHandler.post(() -> {
            for (AuthStateListener authStateChangeListener : mAuthStateListeners) {
                authStateChangeListener.onAuthStateChange(authState);
            }
        });
    }

    /**
     * 消息发送状态监听
     */
    public void notifyMessageSendStateChange(final String msgId, final int messageState) {
        switch (messageState) {
            case ChatMessageListener.MESSAGE_SEND_FAILED:
                getDbService().setMsgSendFailure(msgId);
                break;
            case ChatMessageListener.MESSAGE_SEND_SUCCESS:
                getDbService().setMsgSendSuccess(msgId);
                break;
        }

        mHandler.post(() -> {
            for (ChatMessageListener listener : mChatMessageListeners) {
                listener.onMessageSendStateChange(messageState, msgId);
            }
        });
    }

    private IChatMessageService getDbService() {
        return ChatMessageServiceImp.getInstance(ImSetting.getContext());
    }


    /**
     * 新消息来临
     */
    public void notifyNewMesssage(final ChatMessage message) {
        if (message == null) return;
        NotificationMananger.showNewsNotification(ImSetting.getContext(), message.getFromId());
        bell();
        mHandler.post(() -> {
            if (message != null) {
                ChatMessageServiceImp.getInstance(ImSetting.getContext()).addMessage(message);  // 数据库添加消息
                AppUtils.setBadger(ImSetting.getContext(), ChatMessageServiceImp.getInstance(ImSetting.getContext()).getUnReadMsg());
                for (int i = mChatMessageListeners.size() - 1; i >= 0; i--) {
                    mChatMessageListeners.get(i).onNewMessage(message.getFromUserId(), message, false);
                }
            }
        });
    }

    /**
     * 新朋友发送消息的状态变化
     */
    public void notifyNewFriendSendStateChange(final String toUserId, final ChatMessage message, final int messageState) {
        if (mNewFriendListeners.size() <= 0) {
            return;
        }
        mHandler.post(() -> {
            for (NewFriendListener listener : mNewFriendListeners) {
                listener.onNewFriendSendStateChange(toUserId, message, messageState);
            }
        });
    }


    /*********
     * 收到添加好友
     * @param message
     */
    public void notifyNewFriend(final ChatMessage message) {
        if (message == null) return;
        NotificationMananger.showNewsFriendNotification(ImSetting.getContext(), message.getFromId());
        bell();
        mHandler.post(() -> {
            FriendServiceImp.getInstance(ImSetting.getContext()).addMessage(
                    buildAddFriendMessage(message.getFromUserId(), message.getFromUserName(), message.getContent(), message.getMessageType()));
            boolean hasRead = false;// 是否已经被读了 (如果有类添加)
            for (NewFriendListener listener : mNewFriendListeners) {
                if (listener.onNewFriend(message)) {
                    hasRead = true;
                }
            }
//            if (!hasRead) {
//                Log.e("msg", "新的朋友刷新");
//                int i = NewFriendDao.getInstance().getNewFriendUnRead(message.getOwnerId(), message.getUserId());
//                if (i <= 0) {// 当该新的朋友存在一条未读消息时，不在更新
//                    NewFriendDao.getInstance().markNewFriendUnRead(message.getOwnerId(), message.getUserId());
//                    FriendDao.getInstance().markUserMessageUnRead(loginUserId, Friend.ID_NEW_FRIEND_MESSAGE);
//                }
//                MsgBroadcast.broadcastMsgNumUpdateNewFriend(MyApplication.getInstance());
//            }
//            MsgBroadcast.broadcastMsgUiUpdate(MyApplication.getInstance());
        });
    }

    private void bell() {
        MediaPlayer mediaPlayer = MediaPlayer.create(ImSetting.getContext(), R.raw.msg);
        mediaPlayer.start();
        mediaPlayer.setOnCompletionListener(arg0 -> {
            mediaPlayer.start();
            stopPlay(mediaPlayer);
        });
    }

    private void stopPlay(MediaPlayer mediaPlayer) {
        if (mediaPlayer != null) {
            try {
                mediaPlayer.stop();
            } catch (Exception e) {
                e.printStackTrace();
            }
            mediaPlayer.release();
        }
    }

    /********
     * 收到好友添加请求
     * @param userId
     * @param userName
     * @param status
     * @return
     */
    private Friend buildAddFriendMessage(String userId, String userName, String content, int status) {
        Friend friend = new Friend();
        friend.setUserId(userId);
        friend.setUserName(userName);
        if (status == MessageTypeEnum.HELLO.getType()) {
            friend.setAction("同意");
            friend.setDescribe(content);
            friend.setStatus(Friend.STATUS_11);
        } else if (status == MessageTypeEnum.PASS.getType()) {
            friend.setAction("添加成功");
            friend.setStatus(Friend.STATUS_13);
        } else if (status == MessageTypeEnum.BE_FRIEND.getType()) {
            friend.setAction("成为好友");
            friend.setStatus(Friend.STATUS_21);
        }
        friend.setRead(true);
        friend.setTimeCreate(System.currentTimeMillis());
        return friend;
    }
}
