package com.ruiwenliu.wrapper.util;

import android.content.Context;
import android.content.Intent;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.widget.Toast;

import com.baidu.mapapi.model.LatLng;

import java.lang.ref.WeakReference;
import java.util.List;


/**
 * @author Auser
 * @create 2019-03-26
 * @Describe 地图坐标转换工具类
 */
public class MapUtil {

    private static MapUtil mapUtil;

    private WeakReference<Context> weakReference;

    public MapUtil(Context context) {
        weakReference = new WeakReference<>(context);
    }

    public static MapUtil getInstance(Context context) {
        if (mapUtil == null) {
            mapUtil = new MapUtil(context);
        }

        return mapUtil;
    }


    private Context getContext() {
        return weakReference.get();
    }

    /**
     * BD-09 坐标转换成 GCJ-02 坐标
     * 百度转高德
     */
    public LatLng BD2GCJ(LatLng bd) {
        double x = bd.longitude - 0.0065, y = bd.latitude - 0.006;
        double z = Math.sqrt(x * x + y * y) - 0.00002 * Math.sin(y * Math.PI);
        double theta = Math.atan2(y, x) - 0.000003 * Math.cos(x * Math.PI);

        double lng = z * Math.cos(theta);//lng
        double lat = z * Math.sin(theta);//lat
        return new LatLng(lat, lng);
    }

    /**
     * GCJ-02 坐标转换成 BD-09 坐标
     * 高德转百度
     */
    public LatLng GCJ2BD(LatLng bd) {
        double x = bd.longitude, y = bd.latitude;
        double z = Math.sqrt(x * x + y * y) + 0.00002 * Math.sin(y * Math.PI);
        double theta = Math.atan2(y, x) + 0.000003 * Math.cos(x * Math.PI);
        double tempLon = z * Math.cos(theta) + 0.0065;
        double tempLat = z * Math.sin(theta) + 0.006;
        return new LatLng(tempLat, tempLon);
    }

    /**
     * 跳转百度地图
     *
     * @param latLng
     * @param address
     * @param isTrue  是否需要转换经纬度
     */
    public void goToBaiduMap(LatLng latLng, String address, boolean isTrue) {
        if (isTrue == true) {
            latLng = GCJ2BD(latLng);
        }
        if (!isInstalled(getContext(), "com.baidu.BaiduMap")) {
            Toast.makeText(getContext(), "请先安装百度地图客户端", Toast.LENGTH_SHORT).show();
            return;
        }
        Intent intent = new Intent();
        intent.setData(Uri.parse("baidumap://map/direction?destination=latlng:"
                + latLng.latitude + ","
                + latLng.longitude + "|name:" + address + // 终点
                "&mode=driving" + // 导航路线方式
                "&src=" + getContext().getPackageName()));
        getContext().startActivity(intent); // 启动调用
    }


    /**
     * 跳转高德地图
     * @param latLng
     * @param address
     * @param isTrue
     */
    public void goToGaodeMap(LatLng ststartP, LatLng latLng, String address, boolean isTrue) {

        if (!isInstalled(getContext(), "com.autonavi.minimap")) {
            Toast.makeText(getContext(), "请先安装高德地图客户端", Toast.LENGTH_SHORT).show();
            return;
        }
        LatLng endPoint = isTrue == true ? BD2GCJ(latLng) : latLng;//坐标转换
        LatLng startPoint = isTrue == true ? BD2GCJ(ststartP) : ststartP;
        StringBuffer stringBuffer = new StringBuffer("androidamap://route?sourceApplication=").append("滴房车");
        stringBuffer.append("&dlat=").append(endPoint.latitude)
                .append("&dlon=").append(endPoint.longitude)
                .append("&slat=").append(startPoint.latitude)
                .append("&slon=").append(startPoint.longitude)
                .append("&keywords=" + address)
                .append("&sname=").append("当前位置")
                .append("&dname=").append(address)
                .append("&dev=").append(0)
                .append("&m=").append(0)
                .append("&t=").append(0)
                .append("&showType=").append(1);
        Intent intent = new Intent("android.intent.action.VIEW", Uri.parse(stringBuffer.toString()));
        intent.setPackage("com.autonavi.minimap");
        getContext().startActivity(intent);
    }

    /**
     * 是否安装指定应用
     *
     * @param context
     * @param packageName
     * @return
     */
    private boolean isInstalled(Context context, String packageName) {
        final PackageManager packageManager = context.getPackageManager();
        // 获取所有已安装程序的包信息
        List<PackageInfo> pinfo = packageManager.getInstalledPackages(0);
        for (int i = 0; i < pinfo.size(); i++) {
            if (pinfo.get(i).packageName.equalsIgnoreCase(packageName))
                return true;
        }
        return false;
    }

    public void onDestroy(){
        if(weakReference!=null){
            weakReference.clear();
            weakReference=null;
        }
        mapUtil=null;
    }
}
